<?php

namespace App\Controllers\Admin;

use App\Models\User;
use App\Models\Message;
use App\Models\DailyUsage;
use App\Models\AuditLog;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Config\Database;

class AnalyticsController
{
    private User $userModel;
    private DailyUsage $usageModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->userModel = new User();
        $this->usageModel = new DailyUsage();
        $this->auditLog = new AuditLog();
    }
    
    public function dashboard(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $db = Database::getInstance();
        
        $stmt = $db->query("SELECT COUNT(*) as total FROM users WHERE role = 'client'");
        $totalUsers = $stmt->fetch()['total'];
        
        $stmt = $db->query("SELECT COUNT(*) as total FROM users WHERE role = 'client' AND status = 'active'");
        $activeUsers = $stmt->fetch()['total'];
        
        $stmt = $db->query("SELECT COUNT(*) as total FROM subscriptions WHERE status = 'active'");
        $activeSubscriptions = $stmt->fetch()['total'];
        
        $stmt = $db->query("SELECT COUNT(*) as total FROM whatsapp_numbers WHERE status = 'connected'");
        $connectedNumbers = $stmt->fetch()['total'];
        
        $stmt = $db->query("SELECT COUNT(*) as total FROM messages WHERE DATE(created_at) = CURDATE()");
        $todayMessages = $stmt->fetch()['total'];
        
        $stmt = $db->query("SELECT COUNT(*) as total FROM messages WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
        $monthMessages = $stmt->fetch()['total'];
        
        $stmt = $db->query("SELECT SUM(cost) as total FROM daily_usage WHERE date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)");
        $monthRevenue = $stmt->fetch()['total'] ?? 0;
        
        $stmt = $db->query(
            "SELECT channel, COUNT(*) as count FROM messages 
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) 
             GROUP BY channel"
        );
        $channelStats = $stmt->fetchAll();
        
        Response::success([
            'overview' => [
                'total_users' => (int)$totalUsers,
                'active_users' => (int)$activeUsers,
                'active_subscriptions' => (int)$activeSubscriptions,
                'connected_numbers' => (int)$connectedNumbers,
                'today_messages' => (int)$todayMessages,
                'month_messages' => (int)$monthMessages,
                'month_revenue' => (float)$monthRevenue
            ],
            'channel_stats' => $channelStats
        ]);
    }
    
    public function usage(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $fromDate = $_GET['from_date'] ?? date('Y-m-d', strtotime('-30 days'));
        $toDate = $_GET['to_date'] ?? date('Y-m-d');
        
        $usage = $this->usageModel->getSystemUsage($fromDate, $toDate);
        
        Response::success([
            'usage' => $usage,
            'period' => ['from' => $fromDate, 'to' => $toDate]
        ]);
    }
    
    public function revenue(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $db = Database::getInstance();
        
        $stmt = $db->query(
            "SELECT DATE_FORMAT(created_at, '%Y-%m') as month,
                    SUM(total) as revenue,
                    COUNT(*) as invoices,
                    SUM(CASE WHEN status = 'paid' THEN total ELSE 0 END) as paid
             FROM invoices
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
             GROUP BY DATE_FORMAT(created_at, '%Y-%m')
             ORDER BY month DESC"
        );
        $monthlyRevenue = $stmt->fetchAll();
        
        $stmt = $db->query(
            "SELECT p.name as package_name, COUNT(s.id) as subscribers,
                    SUM(CASE WHEN s.billing_cycle = 'monthly' THEN p.price_monthly ELSE p.price_yearly/12 END) as mrr
             FROM subscriptions s
             JOIN packages p ON s.package_id = p.id
             WHERE s.status = 'active'
             GROUP BY p.id, p.name"
        );
        $packageRevenue = $stmt->fetchAll();
        
        Response::success([
            'monthly_revenue' => $monthlyRevenue,
            'package_revenue' => $packageRevenue
        ]);
    }
    
    public function logs(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(10, (int)($_GET['limit'] ?? 50)));
        $offset = ($page - 1) * $limit;
        
        $filters = [
            'user_id' => $_GET['user_id'] ?? null,
            'action' => $_GET['action'] ?? null,
            'entity_type' => $_GET['entity_type'] ?? null,
            'from_date' => $_GET['from_date'] ?? null,
            'to_date' => $_GET['to_date'] ?? null
        ];
        
        $filters = array_filter($filters);
        
        $logs = $this->auditLog->getAllLogs($limit, $offset, $filters);
        
        Response::success(['logs' => $logs]);
    }
    
    public function topUsers(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $db = Database::getInstance();
        
        $stmt = $db->query(
            "SELECT u.id, u.email, u.name, u.company_name,
                    COUNT(m.id) as message_count,
                    SUM(du.cost) as total_cost
             FROM users u
             LEFT JOIN messages m ON u.id = m.user_id AND m.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
             LEFT JOIN daily_usage du ON u.id = du.user_id AND du.date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
             WHERE u.role = 'client'
             GROUP BY u.id, u.email, u.name, u.company_name
             ORDER BY message_count DESC
             LIMIT 20"
        );
        $topUsers = $stmt->fetchAll();
        
        Response::success(['top_users' => $topUsers]);
    }
    
    public function messageStats(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $db = Database::getInstance();
        
        $stmt = $db->query(
            "SELECT 
                DATE(created_at) as date,
                COUNT(*) as total,
                SUM(CASE WHEN direction = 'outbound' THEN 1 ELSE 0 END) as sent,
                SUM(CASE WHEN direction = 'inbound' THEN 1 ELSE 0 END) as received,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                SUM(CASE WHEN channel = 'meta' THEN 1 ELSE 0 END) as meta,
                SUM(CASE WHEN channel = 'baileys' THEN 1 ELSE 0 END) as baileys
             FROM messages
             WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
             GROUP BY DATE(created_at)
             ORDER BY date DESC"
        );
        $stats = $stmt->fetchAll();
        
        Response::success(['message_stats' => $stats]);
    }
}
