<?php

namespace App\Controllers\Admin;

use App\Models\Package;
use App\Models\AuditLog;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Utils\Validator;
use App\Utils\Helpers;

class PackageController
{
    private Package $packageModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->packageModel = new Package();
        $this->auditLog = new AuditLog();
    }
    
    public function index(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $packages = $this->packageModel->all(100, 0);
        
        foreach ($packages as &$package) {
            $stats = $this->packageModel->getPackageStats($package['id']);
            $package['stats'] = $stats;
        }
        
        Response::success(['packages' => $packages]);
    }
    
    public function show(int $id): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $package = $this->packageModel->find($id);
        
        if (!$package) {
            Response::notFound('Package not found');
        }
        
        $package['stats'] = $this->packageModel->getPackageStats($id);
        
        Response::success(['package' => $package]);
    }
    
    public function store(): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('name')
            ->minLength('name', 2)
            ->required('slug')
            ->validate();
        
        if ($this->packageModel->exists('slug', $data['slug'])) {
            Response::error('Package slug already exists', 400);
        }
        
        $packageId = $this->packageModel->create([
            'name' => $data['name'],
            'slug' => Helpers::slug($data['slug']),
            'description' => $data['description'] ?? null,
            'price_monthly' => $data['price_monthly'] ?? 0,
            'price_yearly' => $data['price_yearly'] ?? 0,
            'messages_limit' => $data['messages_limit'] ?? 1000,
            'whatsapp_numbers_limit' => $data['whatsapp_numbers_limit'] ?? 1,
            'projects_per_number_limit' => $data['projects_per_number_limit'] ?? 1,
            'api_rate_limit' => $data['api_rate_limit'] ?? 60,
            'meta_enabled' => $data['meta_enabled'] ?? 0,
            'baileys_enabled' => $data['baileys_enabled'] ?? 1,
            'webhook_retries' => $data['webhook_retries'] ?? 3,
            'message_retention_days' => $data['message_retention_days'] ?? 30,
            'support_level' => $data['support_level'] ?? 'basic',
            'features' => isset($data['features']) ? json_encode($data['features']) : null,
            'is_active' => $data['is_active'] ?? 1,
            'sort_order' => $data['sort_order'] ?? 0
        ]);
        
        $this->auditLog->log($admin['id'], 'admin.package.created', 'package', $packageId);
        
        $package = $this->packageModel->find($packageId);
        Response::success(['package' => $package], 'Package created', 201);
    }
    
    public function update(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        $data = Helpers::getRequestBody();
        
        $package = $this->packageModel->find($id);
        if (!$package) {
            Response::notFound('Package not found');
        }
        
        $updateData = [];
        
        $allowedFields = [
            'name', 'description', 'price_monthly', 'price_yearly',
            'messages_limit', 'whatsapp_numbers_limit', 'projects_per_number_limit',
            'api_rate_limit', 'meta_enabled', 'baileys_enabled', 'webhook_retries',
            'message_retention_days', 'support_level', 'is_active', 'sort_order'
        ];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                $updateData[$field] = $data[$field];
            }
        }
        
        if (isset($data['features'])) {
            $updateData['features'] = json_encode($data['features']);
        }
        
        if (!empty($updateData)) {
            $this->packageModel->update($id, $updateData);
            $this->auditLog->log($admin['id'], 'admin.package.updated', 'package', $id, $package, $updateData);
        }
        
        $package = $this->packageModel->find($id);
        Response::success(['package' => $package], 'Package updated');
    }
    
    public function delete(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        $package = $this->packageModel->find($id);
        if (!$package) {
            Response::notFound('Package not found');
        }
        
        if ($package['is_default']) {
            Response::error('Cannot delete default package', 400);
        }
        
        $stats = $this->packageModel->getPackageStats($id);
        if ($stats['subscribers'] > 0) {
            Response::error('Cannot delete package with active subscribers', 400);
        }
        
        $this->packageModel->delete($id);
        $this->auditLog->log($admin['id'], 'admin.package.deleted', 'package', $id);
        
        Response::success(null, 'Package deleted');
    }
    
    public function setDefault(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        $package = $this->packageModel->find($id);
        if (!$package) {
            Response::notFound('Package not found');
        }
        
        $this->packageModel->setDefault($id);
        $this->auditLog->log($admin['id'], 'admin.package.set_default', 'package', $id);
        
        Response::success(null, 'Default package updated');
    }
}
