<?php

namespace App\Controllers\Admin;

use App\Models\SystemSetting;
use App\Models\RouteSetting;
use App\Models\AuditLog;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Utils\Helpers;

class SettingsController
{
    private SystemSetting $settingModel;
    private RouteSetting $routeSettingModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->settingModel = new SystemSetting();
        $this->routeSettingModel = new RouteSetting();
        $this->auditLog = new AuditLog();
    }
    
    public function index(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $settings = $this->settingModel->getAllSettings();
        Response::success(['settings' => $settings]);
    }
    
    public function update(): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        $data = Helpers::getRequestBody();
        
        foreach ($data as $key => $value) {
            $this->settingModel->set($key, $value);
        }
        
        $this->auditLog->log($admin['id'], 'admin.settings.updated', 'system_settings', null, null, $data);
        
        Response::success(null, 'Settings updated');
    }
    
    public function routes(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $routes = $this->routeSettingModel->getAllRoutes();
        Response::success(['routes' => $routes]);
    }
    
    public function updateRoute(): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        $data = Helpers::getRequestBody();
        
        $routeName = $data['route_name'] ?? null;
        
        if (!$routeName) {
            Response::error('Route name required', 400);
        }
        
        $route = $this->routeSettingModel->getRoute($routeName);
        if (!$route) {
            Response::error('Route not found', 404);
        }
        
        $updateData = [];
        
        if (isset($data['is_enabled'])) {
            $updateData['is_enabled'] = $data['is_enabled'] ? 1 : 0;
        }
        
        if (isset($data['cost_per_message'])) {
            $updateData['cost_per_message'] = (float)$data['cost_per_message'];
        }
        
        if (isset($data['priority'])) {
            $updateData['priority'] = (int)$data['priority'];
        }
        
        if (isset($data['settings'])) {
            $updateData['settings'] = json_encode($data['settings']);
        }
        
        if (!empty($updateData)) {
            $this->routeSettingModel->update($route['id'], $updateData);
            $this->auditLog->log($admin['id'], 'admin.route.updated', 'route_settings', $route['id'], $route, $updateData);
        }
        
        $route = $this->routeSettingModel->getRoute($routeName);
        Response::success(['route' => $route], 'Route updated');
    }
    
    public function enableRoute(string $routeName): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        if (!$this->routeSettingModel->enableRoute($routeName)) {
            Response::error('Route not found', 404);
        }
        
        $this->auditLog->log($admin['id'], 'admin.route.enabled', 'route_settings', null, null, ['route' => $routeName]);
        
        Response::success(null, 'Route enabled');
    }
    
    public function disableRoute(string $routeName): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        if (!$this->routeSettingModel->disableRoute($routeName)) {
            Response::error('Route not found', 404);
        }
        
        $this->auditLog->log($admin['id'], 'admin.route.disabled', 'route_settings', null, null, ['route' => $routeName]);
        
        Response::success(null, 'Route disabled');
    }
}
