<?php

namespace App\Controllers\Admin;

use App\Models\User;
use App\Models\Subscription;
use App\Models\WhatsappNumber;
use App\Models\AuditLog;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Utils\Validator;
use App\Utils\Helpers;

class UserController
{
    private User $userModel;
    private Subscription $subscriptionModel;
    private WhatsappNumber $numberModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->userModel = new User();
        $this->subscriptionModel = new Subscription();
        $this->numberModel = new WhatsappNumber();
        $this->auditLog = new AuditLog();
    }
    
    public function index(): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(10, (int)($_GET['limit'] ?? 50)));
        $offset = ($page - 1) * $limit;
        
        $role = $_GET['role'] ?? null;
        $status = $_GET['status'] ?? null;
        
        $users = $this->userModel->getAllUsers($limit, $offset, $role, $status);
        $total = $this->userModel->countUsers($role, $status);
        
        Response::success([
            'users' => $users,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
    }
    
    public function show(int $id): void
    {
        AuthMiddleware::authenticateAdmin();
        
        $user = $this->userModel->getUserWithSubscription($id);
        
        if (!$user) {
            Response::notFound('User not found');
        }
        
        $numbers = $this->numberModel->getUserNumbers($id);
        $user['whatsapp_numbers'] = $numbers;
        
        Response::success(['user' => $user]);
    }
    
    public function store(): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('email')
            ->email('email')
            ->required('name')
            ->minLength('name', 2)
            ->validate();
        
        if ($this->userModel->exists('email', $data['email'])) {
            Response::error('Email already registered', 400);
        }
        
        $password = $data['password'] ?? Helpers::generateToken(8);
        
        $userId = $this->userModel->createUser([
            'email' => $data['email'],
            'password' => $password,
            'name' => $data['name'],
            'phone' => $data['phone'] ?? null,
            'company_name' => $data['company_name'] ?? null,
            'role' => $data['role'] ?? 'client',
            'status' => $data['status'] ?? 'active'
        ]);
        
        if (isset($data['package_id'])) {
            $this->subscriptionModel->createSubscription($userId, $data['package_id']);
        }
        
        $this->auditLog->log($admin['id'], 'admin.user.created', 'user', $userId);
        
        $user = $this->userModel->find($userId);
        Response::success(['user' => $user, 'password' => $password], 'User created', 201);
    }
    
    public function update(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        $data = Helpers::getRequestBody();
        
        $user = $this->userModel->find($id);
        if (!$user) {
            Response::notFound('User not found');
        }
        
        $updateData = [];
        
        if (isset($data['name'])) {
            $updateData['name'] = $data['name'];
        }
        if (isset($data['phone'])) {
            $updateData['phone'] = $data['phone'];
        }
        if (isset($data['company_name'])) {
            $updateData['company_name'] = $data['company_name'];
        }
        if (isset($data['status'])) {
            $updateData['status'] = $data['status'];
        }
        if (isset($data['role']) && $admin['role'] === 'superadmin') {
            $updateData['role'] = $data['role'];
        }
        
        if (!empty($updateData)) {
            $this->userModel->update($id, $updateData);
            $this->auditLog->log($admin['id'], 'admin.user.updated', 'user', $id, $user, $updateData);
        }
        
        $user = $this->userModel->find($id);
        Response::success(['user' => $user], 'User updated');
    }
    
    public function suspend(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        $user = $this->userModel->find($id);
        if (!$user) {
            Response::notFound('User not found');
        }
        
        if ($user['role'] === 'superadmin') {
            Response::forbidden('Cannot suspend superadmin');
        }
        
        $this->userModel->suspendUser($id);
        $this->auditLog->log($admin['id'], 'admin.user.suspended', 'user', $id);
        
        Response::success(null, 'User suspended');
    }
    
    public function ban(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        $user = $this->userModel->find($id);
        if (!$user) {
            Response::notFound('User not found');
        }
        
        if ($user['role'] === 'superadmin') {
            Response::forbidden('Cannot ban superadmin');
        }
        
        $this->userModel->banUser($id);
        $this->auditLog->log($admin['id'], 'admin.user.banned', 'user', $id);
        
        Response::success(null, 'User banned');
    }
    
    public function activate(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        $user = $this->userModel->find($id);
        if (!$user) {
            Response::notFound('User not found');
        }
        
        $this->userModel->activateUser($id);
        $this->auditLog->log($admin['id'], 'admin.user.activated', 'user', $id);
        
        Response::success(null, 'User activated');
    }
    
    public function resetPassword(int $id): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        $user = $this->userModel->find($id);
        if (!$user) {
            Response::notFound('User not found');
        }
        
        $newPassword = Helpers::generateToken(8);
        $this->userModel->updatePassword($id, $newPassword);
        $this->auditLog->log($admin['id'], 'admin.user.password_reset', 'user', $id);
        
        Response::success(['password' => $newPassword], 'Password reset');
    }
    
    public function disconnectWhatsapp(int $userId, int $numberId): void
    {
        $admin = AuthMiddleware::authenticateAdmin();
        
        $number = $this->numberModel->findUserNumber($userId, $numberId);
        if (!$number) {
            Response::notFound('WhatsApp number not found');
        }
        
        $gatewayService = new \App\Services\GatewayService();
        $gatewayService->disconnectNumber($numberId);
        
        $this->numberModel->setDisconnected($numberId);
        $this->auditLog->log($admin['id'], 'admin.whatsapp.disconnected', 'whatsapp_number', $numberId);
        
        Response::success(null, 'WhatsApp disconnected');
    }
}
