<?php

namespace App\Controllers;

use App\Models\ApiKey;
use App\Models\Project;
use App\Models\AuditLog;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Utils\Validator;
use App\Utils\Helpers;

class ApiKeyController
{
    private ApiKey $apiKeyModel;
    private Project $projectModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->apiKeyModel = new ApiKey();
        $this->projectModel = new Project();
        $this->auditLog = new AuditLog();
    }
    
    public function index(): void
    {
        $user = AuthMiddleware::authenticate();
        $apiKeys = $this->apiKeyModel->getUserApiKeys($user['id']);
        Response::success(['api_keys' => $apiKeys]);
    }
    
    public function show(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        $apiKey = $this->apiKeyModel->findUserApiKey($user['id'], $id);
        
        if (!$apiKey) {
            Response::notFound('API key not found');
        }
        
        unset($apiKey['api_secret']);
        Response::success(['api_key' => $apiKey]);
    }
    
    public function store(): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('project_id')
            ->integer('project_id')
            ->required('name')
            ->minLength('name', 2)
            ->validate();
        
        $project = $this->projectModel->findUserProject($user['id'], $data['project_id']);
        if (!$project) {
            Response::error('Project not found', 404);
        }
        
        $result = $this->apiKeyModel->createApiKey(
            $user['id'],
            $data['project_id'],
            $data['name']
        );
        
        $this->auditLog->log($user['id'], 'api_key.created', 'api_key', $result['id']);
        
        Response::success([
            'id' => $result['id'],
            'api_key' => $result['api_key'],
            'api_secret' => $result['api_secret'],
            'message' => 'Save your API secret now. It will not be shown again.'
        ], 'API key created successfully', 201);
    }
    
    public function update(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        $apiKey = $this->apiKeyModel->findUserApiKey($user['id'], $id);
        if (!$apiKey) {
            Response::notFound('API key not found');
        }
        
        $updateData = [];
        
        if (isset($data['name'])) {
            Validator::make($data)->minLength('name', 2)->validate();
            $updateData['name'] = $data['name'];
        }
        
        if (isset($data['rate_limit'])) {
            $updateData['rate_limit'] = min(1000, max(1, (int)$data['rate_limit']));
        }
        
        if (isset($data['expires_at'])) {
            $updateData['expires_at'] = $data['expires_at'];
        }
        
        if (!empty($updateData)) {
            $this->apiKeyModel->update($id, $updateData);
            $this->auditLog->log($user['id'], 'api_key.updated', 'api_key', $id);
        }
        
        $apiKey = $this->apiKeyModel->findUserApiKey($user['id'], $id);
        unset($apiKey['api_secret']);
        Response::success(['api_key' => $apiKey], 'API key updated');
    }
    
    public function delete(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $apiKey = $this->apiKeyModel->findUserApiKey($user['id'], $id);
        if (!$apiKey) {
            Response::notFound('API key not found');
        }
        
        $this->apiKeyModel->revokeApiKey($id);
        $this->auditLog->log($user['id'], 'api_key.revoked', 'api_key', $id);
        
        Response::success(null, 'API key revoked');
    }
    
    public function regenerateSecret(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $apiKey = $this->apiKeyModel->findUserApiKey($user['id'], $id);
        if (!$apiKey) {
            Response::notFound('API key not found');
        }
        
        $newSecret = $this->apiKeyModel->regenerateSecret($id);
        $this->auditLog->log($user['id'], 'api_key.secret_regenerated', 'api_key', $id);
        
        Response::success([
            'api_secret' => $newSecret,
            'message' => 'Save your new API secret now. It will not be shown again.'
        ], 'API secret regenerated');
    }
    
    public function setIpWhitelist(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        $apiKey = $this->apiKeyModel->findUserApiKey($user['id'], $id);
        if (!$apiKey) {
            Response::notFound('API key not found');
        }
        
        $ips = $data['ips'] ?? [];
        
        if (!is_array($ips)) {
            Response::error('IPs must be an array', 400);
        }
        
        foreach ($ips as $ip) {
            if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                Response::error("Invalid IP address: {$ip}", 400);
            }
        }
        
        $this->apiKeyModel->setIpWhitelist($id, $ips);
        $this->auditLog->log($user['id'], 'api_key.ip_whitelist_updated', 'api_key', $id);
        
        Response::success(['ip_whitelist' => $ips], 'IP whitelist updated');
    }
}
