<?php

namespace App\Controllers;

use App\Services\AuthService;
use App\Middleware\AuthMiddleware;
use App\Middleware\RateLimitMiddleware;
use App\Utils\Response;
use App\Utils\Validator;
use App\Utils\Helpers;

class AuthController
{
    private AuthService $authService;
    
    public function __construct()
    {
        $this->authService = new AuthService();
    }
    
    public function register(): void
    {
        RateLimitMiddleware::check(10, 60);
        
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('email')
            ->email('email')
            ->required('password')
            ->minLength('password', 8)
            ->required('name')
            ->minLength('name', 2)
            ->validate();
        
        $result = $this->authService->register($data);
        
        Response::success($result, 'Registration successful', 201);
    }
    
    public function login(): void
    {
        RateLimitMiddleware::check(10, 60);
        
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('email')
            ->email('email')
            ->required('password')
            ->validate();
        
        $result = $this->authService->login($data['email'], $data['password']);
        
        Response::success($result, 'Login successful');
    }
    
    public function googleAuthUrl(): void
    {
        $url = $this->authService->getGoogleAuthUrl();
        Response::success(['url' => $url]);
    }
    
    public function googleCallback(): void
    {
        $code = $_GET['code'] ?? null;
        
        if (!$code) {
            Response::error('Authorization code required', 400);
        }
        
        $result = $this->authService->googleAuth($code);
        
        Response::success($result, 'Google authentication successful');
    }
    
    public function logout(): void
    {
        $user = AuthMiddleware::authenticate();
        $this->authService->logout($user['id']);
        Response::success(null, 'Logged out successfully');
    }
    
    public function me(): void
    {
        $user = AuthMiddleware::authenticate();
        Response::success(['user' => $user]);
    }
    
    public function refreshToken(): void
    {
        $user = AuthMiddleware::authenticate();
        $result = $this->authService->refreshToken($user);
        Response::success($result, 'Token refreshed');
    }
    
    public function updateProfile(): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        $validator = Validator::make($data);
        
        if (isset($data['name'])) {
            $validator->minLength('name', 2);
        }
        if (isset($data['phone'])) {
            $validator->phone('phone');
        }
        
        $validator->validate();
        
        $userModel = new \App\Models\User();
        $updateData = array_intersect_key($data, array_flip([
            'name', 'phone', 'company_name', 'timezone', 'language'
        ]));
        
        if (!empty($updateData)) {
            $userModel->update($user['id'], $updateData);
        }
        
        $updatedUser = $userModel->find($user['id']);
        Response::success(['user' => $updatedUser], 'Profile updated');
    }
    
    public function changePassword(): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('current_password')
            ->required('new_password')
            ->minLength('new_password', 8)
            ->confirmed('new_password')
            ->validate();
        
        $userModel = new \App\Models\User();
        $fullUser = $userModel->findByEmail($user['email']);
        
        if (!$fullUser['password']) {
            Response::error('Password change not available for OAuth accounts', 400);
        }
        
        if (!Helpers::verifyPassword($data['current_password'], $fullUser['password'])) {
            Response::error('Current password is incorrect', 400);
        }
        
        $userModel->updatePassword($user['id'], $data['new_password']);
        
        Response::success(null, 'Password changed successfully');
    }
}
