<?php

namespace App\Controllers;

use App\Models\Package;
use App\Models\Subscription;
use App\Models\DailyUsage;
use App\Models\AuditLog;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Utils\Helpers;
use App\Config\Database;

class BillingController
{
    private Package $packageModel;
    private Subscription $subscriptionModel;
    private DailyUsage $usageModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->packageModel = new Package();
        $this->subscriptionModel = new Subscription();
        $this->usageModel = new DailyUsage();
        $this->auditLog = new AuditLog();
    }
    
    public function packages(): void
    {
        $packages = $this->packageModel->getActivePackages();
        Response::success(['packages' => $packages]);
    }
    
    public function currentSubscription(): void
    {
        $user = AuthMiddleware::authenticate();
        $subscription = $this->subscriptionModel->getActiveSubscription($user['id']);
        
        if (!$subscription) {
            Response::success(['subscription' => null, 'message' => 'No active subscription']);
            return;
        }
        
        $quota = $this->subscriptionModel->checkQuota($user['id']);
        $subscription['quota'] = $quota;
        
        Response::success(['subscription' => $subscription]);
    }
    
    public function subscribe(): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        $packageId = $data['package_id'] ?? null;
        $billingCycle = $data['billing_cycle'] ?? 'monthly';
        
        if (!$packageId) {
            Response::error('Package ID required', 400);
        }
        
        $package = $this->packageModel->find($packageId);
        if (!$package || !$package['is_active']) {
            Response::error('Package not found or inactive', 404);
        }
        
        if (!in_array($billingCycle, ['monthly', 'yearly'])) {
            $billingCycle = 'monthly';
        }
        
        $subscriptionId = $this->subscriptionModel->createSubscription(
            $user['id'],
            $packageId,
            $billingCycle
        );
        
        $this->auditLog->log($user['id'], 'subscription.created', 'subscription', $subscriptionId, null, [
            'package_id' => $packageId,
            'billing_cycle' => $billingCycle
        ]);
        
        $subscription = $this->subscriptionModel->getActiveSubscription($user['id']);
        
        Response::success(['subscription' => $subscription], 'Subscription activated', 201);
    }
    
    public function cancelSubscription(): void
    {
        $user = AuthMiddleware::authenticate();
        
        $subscription = $this->subscriptionModel->getActiveSubscription($user['id']);
        if (!$subscription) {
            Response::error('No active subscription to cancel', 400);
        }
        
        $this->subscriptionModel->cancelActiveSubscriptions($user['id']);
        $this->auditLog->log($user['id'], 'subscription.cancelled', 'subscription', $subscription['id']);
        
        Response::success(null, 'Subscription cancelled');
    }
    
    public function usage(): void
    {
        $user = AuthMiddleware::authenticate();
        
        $fromDate = $_GET['from_date'] ?? date('Y-m-01');
        $toDate = $_GET['to_date'] ?? date('Y-m-d');
        
        $usage = $this->usageModel->getUserUsage($user['id'], $fromDate, $toDate);
        $total = $this->usageModel->getTotalUsage($user['id']);
        
        Response::success([
            'usage' => $usage,
            'total' => $total,
            'period' => [
                'from' => $fromDate,
                'to' => $toDate
            ]
        ]);
    }
    
    public function quota(): void
    {
        $user = AuthMiddleware::authenticate();
        $quota = $this->subscriptionModel->checkQuota($user['id']);
        Response::success(['quota' => $quota]);
    }
    
    public function credits(): void
    {
        $user = AuthMiddleware::authenticate();
        
        $db = Database::getInstance();
        $stmt = $db->prepare("SELECT balance FROM credits WHERE user_id = ?");
        $stmt->execute([$user['id']]);
        $credits = $stmt->fetch();
        
        $stmt = $db->prepare(
            "SELECT * FROM credit_transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 50"
        );
        $stmt->execute([$user['id']]);
        $transactions = $stmt->fetchAll();
        
        Response::success([
            'balance' => $credits ? (float)$credits['balance'] : 0,
            'transactions' => $transactions
        ]);
    }
    
    public function invoices(): void
    {
        $user = AuthMiddleware::authenticate();
        
        $db = Database::getInstance();
        $stmt = $db->prepare(
            "SELECT * FROM invoices WHERE user_id = ? ORDER BY created_at DESC LIMIT 50"
        );
        $stmt->execute([$user['id']]);
        $invoices = $stmt->fetchAll();
        
        Response::success(['invoices' => $invoices]);
    }
    
    public function invoiceDetails(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $db = Database::getInstance();
        $stmt = $db->prepare("SELECT * FROM invoices WHERE id = ? AND user_id = ?");
        $stmt->execute([$id, $user['id']]);
        $invoice = $stmt->fetch();
        
        if (!$invoice) {
            Response::notFound('Invoice not found');
        }
        
        Response::success(['invoice' => $invoice]);
    }
}
