<?php

namespace App\Controllers;

use App\Models\Message;
use App\Models\Project;
use App\Middleware\AuthMiddleware;
use App\Middleware\RateLimitMiddleware;
use App\Services\GatewayService;
use App\Utils\Response;
use App\Utils\Validator;
use App\Utils\Helpers;

class MessageController
{
    private Message $messageModel;
    private Project $projectModel;
    private GatewayService $gatewayService;
    
    public function __construct()
    {
        $this->messageModel = new Message();
        $this->projectModel = new Project();
        $this->gatewayService = new GatewayService();
    }
    
    public function sendMessage(): void
    {
        $apiKeyData = AuthMiddleware::authenticateApi();
        RateLimitMiddleware::checkApiKey($apiKeyData);
        
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('to')
            ->phone('to')
            ->required('message')
            ->validate();
        
        $result = $this->gatewayService->sendMessage($apiKeyData, [
            'to' => $data['to'],
            'message' => $data['message'],
            'type' => $data['type'] ?? 'text'
        ]);
        
        Response::success($result);
    }
    
    public function sendMedia(): void
    {
        $apiKeyData = AuthMiddleware::authenticateApi();
        RateLimitMiddleware::checkApiKey($apiKeyData);
        
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('to')
            ->phone('to')
            ->required('media_url')
            ->url('media_url')
            ->validate();
        
        $result = $this->gatewayService->sendMedia($apiKeyData, [
            'to' => $data['to'],
            'media_url' => $data['media_url'],
            'caption' => $data['caption'] ?? null,
            'type' => $data['type'] ?? 'image',
            'filename' => $data['filename'] ?? null
        ]);
        
        Response::success($result);
    }
    
    public function requestQr(): void
    {
        $apiKeyData = AuthMiddleware::authenticateApi();
        
        $result = $this->gatewayService->requestQr(
            $apiKeyData['user_id'],
            $apiKeyData['whatsapp_number_id']
        );
        
        Response::success($result);
    }
    
    public function qrStatus(): void
    {
        $token = $_GET['token'] ?? null;
        
        if (!$token) {
            Response::error('QR token required', 400);
        }
        
        $result = $this->gatewayService->getQrStatus($token);
        Response::success($result);
    }
    
    public function index(): void
    {
        $user = AuthMiddleware::authenticate();
        
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(10, (int)($_GET['limit'] ?? 50)));
        $offset = ($page - 1) * $limit;
        
        $filters = [
            'project_id' => $_GET['project_id'] ?? null,
            'direction' => $_GET['direction'] ?? null,
            'status' => $_GET['status'] ?? null,
            'channel' => $_GET['channel'] ?? null,
            'from_date' => $_GET['from_date'] ?? null,
            'to_date' => $_GET['to_date'] ?? null,
            'search' => $_GET['search'] ?? null
        ];
        
        $filters = array_filter($filters);
        
        $messages = $this->messageModel->getUserMessages($user['id'], $limit, $offset, $filters);
        $total = $this->messageModel->countUserMessages($user['id'], $filters);
        
        Response::success([
            'messages' => $messages,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
    }
    
    public function show(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $message = $this->messageModel->find($id);
        
        if (!$message || $message['user_id'] !== $user['id']) {
            Response::notFound('Message not found');
        }
        
        Response::success(['message' => $message]);
    }
    
    public function conversation(int $projectId): void
    {
        $user = AuthMiddleware::authenticate();
        
        $project = $this->projectModel->findUserProject($user['id'], $projectId);
        if (!$project) {
            Response::notFound('Project not found');
        }
        
        $contactNumber = $_GET['contact'] ?? null;
        if (!$contactNumber) {
            Response::error('Contact number required', 400);
        }
        
        $limit = min(500, max(10, (int)($_GET['limit'] ?? 100)));
        
        $messages = $this->messageModel->getConversation($projectId, $contactNumber, $limit);
        
        Response::success(['messages' => $messages]);
    }
    
    public function stats(): void
    {
        $user = AuthMiddleware::authenticate();
        
        $period = $_GET['period'] ?? 'day';
        if (!in_array($period, ['hour', 'day', 'week', 'month'])) {
            $period = 'day';
        }
        
        $stats = $this->messageModel->getMessageStats($user['id'], $period);
        
        Response::success(['stats' => $stats]);
    }
    
    public function export(): void
    {
        $user = AuthMiddleware::authenticate();
        
        $filters = [
            'project_id' => $_GET['project_id'] ?? null,
            'direction' => $_GET['direction'] ?? null,
            'status' => $_GET['status'] ?? null,
            'channel' => $_GET['channel'] ?? null,
            'from_date' => $_GET['from_date'] ?? null,
            'to_date' => $_GET['to_date'] ?? null
        ];
        
        $filters = array_filter($filters);
        
        $messages = $this->messageModel->exportMessages($user['id'], $filters);
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="messages_export_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        fputcsv($output, [
            'ID', 'Direction', 'Channel', 'From', 'To', 'Type', 'Content',
            'Status', 'Cost', 'Created At', 'Sent At', 'Delivered At', 'Read At'
        ]);
        
        foreach ($messages as $message) {
            fputcsv($output, [
                $message['uuid'],
                $message['direction'],
                $message['channel'],
                $message['from_number'],
                $message['to_number'],
                $message['message_type'],
                $message['content'],
                $message['status'],
                $message['cost'],
                $message['created_at'],
                $message['sent_at'],
                $message['delivered_at'],
                $message['read_at']
            ]);
        }
        
        fclose($output);
        exit;
    }
}
