<?php

namespace App\Controllers;

use App\Models\Project;
use App\Models\WhatsappNumber;
use App\Models\Subscription;
use App\Models\AuditLog;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Utils\Validator;
use App\Utils\Helpers;

class ProjectController
{
    private Project $projectModel;
    private WhatsappNumber $numberModel;
    private Subscription $subscriptionModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->projectModel = new Project();
        $this->numberModel = new WhatsappNumber();
        $this->subscriptionModel = new Subscription();
        $this->auditLog = new AuditLog();
    }
    
    public function index(): void
    {
        $user = AuthMiddleware::authenticate();
        $projects = $this->projectModel->getUserProjects($user['id']);
        Response::success(['projects' => $projects]);
    }
    
    public function show(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        $project = $this->projectModel->findUserProject($user['id'], $id);
        
        if (!$project) {
            Response::notFound('Project not found');
        }
        
        Response::success(['project' => $project]);
    }
    
    public function store(): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('name')
            ->minLength('name', 2)
            ->required('whatsapp_number_id')
            ->integer('whatsapp_number_id')
            ->validate();
        
        $number = $this->numberModel->findUserNumber($user['id'], $data['whatsapp_number_id']);
        if (!$number) {
            Response::error('WhatsApp number not found', 404);
        }
        
        $subscription = $this->subscriptionModel->getActiveSubscription($user['id']);
        if (!$subscription) {
            Response::error('No active subscription', 403);
        }
        
        $projectsForNumber = $this->projectModel->countProjectsForNumber($data['whatsapp_number_id']);
        if ($projectsForNumber >= $subscription['projects_per_number_limit']) {
            Response::error('Projects limit reached for this WhatsApp number', 403);
        }
        
        $projectId = $this->projectModel->createProject([
            'user_id' => $user['id'],
            'whatsapp_number_id' => $data['whatsapp_number_id'],
            'name' => $data['name'],
            'description' => $data['description'] ?? null,
            'webhook_url' => $data['webhook_url'] ?? null,
            'webhook_enabled' => isset($data['webhook_url']) ? 1 : 0
        ]);
        
        $this->auditLog->log($user['id'], 'project.created', 'project', $projectId);
        
        $project = $this->projectModel->findUserProject($user['id'], $projectId);
        Response::success(['project' => $project], 'Project created successfully', 201);
    }
    
    public function update(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        $project = $this->projectModel->findUserProject($user['id'], $id);
        if (!$project) {
            Response::notFound('Project not found');
        }
        
        $updateData = [];
        
        if (isset($data['name'])) {
            Validator::make($data)->minLength('name', 2)->validate();
            $updateData['name'] = $data['name'];
        }
        
        if (isset($data['description'])) {
            $updateData['description'] = $data['description'];
        }
        
        if (isset($data['webhook_url'])) {
            Validator::make($data)->url('webhook_url')->validate();
            $updateData['webhook_url'] = $data['webhook_url'];
        }
        
        if (isset($data['webhook_enabled'])) {
            $updateData['webhook_enabled'] = $data['webhook_enabled'] ? 1 : 0;
        }
        
        if (isset($data['webhook_retry_count'])) {
            $updateData['webhook_retry_count'] = min(10, max(0, (int)$data['webhook_retry_count']));
        }
        
        if (isset($data['webhook_retry_delay'])) {
            $updateData['webhook_retry_delay'] = min(3600, max(10, (int)$data['webhook_retry_delay']));
        }
        
        if (!empty($updateData)) {
            $this->projectModel->update($id, $updateData);
            $this->auditLog->log($user['id'], 'project.updated', 'project', $id);
        }
        
        $project = $this->projectModel->findUserProject($user['id'], $id);
        Response::success(['project' => $project], 'Project updated');
    }
    
    public function delete(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $project = $this->projectModel->findUserProject($user['id'], $id);
        if (!$project) {
            Response::notFound('Project not found');
        }
        
        $this->projectModel->update($id, ['is_active' => 0]);
        $this->auditLog->log($user['id'], 'project.deleted', 'project', $id);
        
        Response::success(null, 'Project deleted');
    }
    
    public function regenerateWebhookSecret(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $project = $this->projectModel->findUserProject($user['id'], $id);
        if (!$project) {
            Response::notFound('Project not found');
        }
        
        $newSecret = $this->projectModel->regenerateWebhookSecret($id);
        $this->auditLog->log($user['id'], 'project.webhook_secret_regenerated', 'project', $id);
        
        Response::success(['webhook_secret' => $newSecret], 'Webhook secret regenerated');
    }
    
    public function getWithApiKeys(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $project = $this->projectModel->findUserProject($user['id'], $id);
        if (!$project) {
            Response::notFound('Project not found');
        }
        
        $projectWithKeys = $this->projectModel->getProjectWithApiKeys($id);
        Response::success(['project' => $projectWithKeys]);
    }
}
