<?php

namespace App\Controllers;

use App\Models\Message;
use App\Models\Project;
use App\Models\WhatsappNumber;
use App\Models\WebhookLog;
use App\Models\DailyUsage;
use App\Models\Subscription;
use App\Services\WebhookService;
use App\Utils\Response;
use App\Utils\Helpers;
use App\Config\App;

class WebhookController
{
    private Message $messageModel;
    private Project $projectModel;
    private WhatsappNumber $numberModel;
    private WebhookService $webhookService;
    private DailyUsage $usageModel;
    private Subscription $subscriptionModel;
    
    public function __construct()
    {
        $this->messageModel = new Message();
        $this->projectModel = new Project();
        $this->numberModel = new WhatsappNumber();
        $this->webhookService = new WebhookService();
        $this->usageModel = new DailyUsage();
        $this->subscriptionModel = new Subscription();
    }
    
    public function handleMeta(): void
    {
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            $this->verifyMetaWebhook();
            return;
        }
        
        $this->validateGatewaySecret();
        
        $payload = Helpers::getRequestBody();
        
        if (empty($payload['entry'])) {
            Response::success(['status' => 'ignored']);
            return;
        }
        
        foreach ($payload['entry'] as $entry) {
            foreach ($entry['changes'] ?? [] as $change) {
                if ($change['field'] === 'messages') {
                    $this->processMetaMessage($change['value']);
                }
            }
        }
        
        Response::success(['status' => 'processed']);
    }
    
    public function handleBaileys(): void
    {
        $this->validateGatewaySecret();
        
        $payload = Helpers::getRequestBody();
        
        $eventType = $payload['event'] ?? null;
        
        switch ($eventType) {
            case 'message':
                $this->processBaileysMessage($payload);
                break;
            case 'message.status':
                $this->processMessageStatus($payload);
                break;
            case 'connection.update':
                $this->processConnectionUpdate($payload);
                break;
            case 'qr.update':
                $this->processQrUpdate($payload);
                break;
            default:
                Response::success(['status' => 'ignored']);
        }
    }
    
    private function verifyMetaWebhook(): void
    {
        $mode = $_GET['hub_mode'] ?? null;
        $token = $_GET['hub_verify_token'] ?? null;
        $challenge = $_GET['hub_challenge'] ?? null;
        
        $verifyToken = $_ENV['META_WEBHOOK_VERIFY_TOKEN'] ?? 'whatsapp_gateway_verify';
        
        if ($mode === 'subscribe' && $token === $verifyToken) {
            echo $challenge;
            exit;
        }
        
        Response::error('Verification failed', 403);
    }
    
    private function validateGatewaySecret(): void
    {
        $secret = $_SERVER['HTTP_X_GATEWAY_SECRET'] ?? '';
        
        if (!hash_equals(App::getGatewaySecret(), $secret)) {
            Response::unauthorized('Invalid gateway secret');
        }
    }
    
    private function processMetaMessage(array $value): void
    {
        $metadata = $value['metadata'] ?? [];
        $phoneNumberId = $metadata['phone_number_id'] ?? null;
        
        if (!$phoneNumberId) {
            return;
        }
        
        $number = $this->numberModel->findBy('meta_phone_number_id', $phoneNumberId);
        if (!$number) {
            return;
        }
        
        $this->numberModel->updateLastActive($number['id']);
        
        foreach ($value['messages'] ?? [] as $msg) {
            $this->storeInboundMessage($number, $msg, 'meta');
        }
        
        foreach ($value['statuses'] ?? [] as $status) {
            $this->updateMessageStatus($status);
        }
    }
    
    private function processBaileysMessage(array $payload): void
    {
        $whatsappNumberId = $payload['whatsapp_number_id'] ?? null;
        
        if (!$whatsappNumberId) {
            return;
        }
        
        $number = $this->numberModel->find($whatsappNumberId);
        if (!$number) {
            return;
        }
        
        $this->numberModel->updateLastActive($number['id']);
        
        $this->storeInboundMessage($number, $payload['message'], 'baileys');
    }
    
    private function storeInboundMessage(array $number, array $msg, string $channel): void
    {
        $projects = $this->projectModel->getProjectsByNumber($number['id']);
        
        if (empty($projects)) {
            return;
        }
        
        $project = $projects[0];
        
        $fromNumber = $channel === 'meta' 
            ? '+' . ($msg['from'] ?? '') 
            : ($msg['from'] ?? '');
        
        $messageType = $this->determineMessageType($msg, $channel);
        $content = $this->extractContent($msg, $channel);
        $mediaUrl = $this->extractMediaUrl($msg, $channel);
        
        $messageId = $this->messageModel->createMessage([
            'user_id' => $number['user_id'],
            'project_id' => $project['id'],
            'whatsapp_number_id' => $number['id'],
            'direction' => 'inbound',
            'channel' => $channel,
            'from_number' => $fromNumber,
            'to_number' => $number['phone_number'],
            'message_type' => $messageType,
            'content' => $content,
            'media_url' => $mediaUrl,
            'external_id' => $msg['id'] ?? null,
            'status' => 'delivered'
        ]);
        
        $this->usageModel->incrementStat($number['user_id'], $project['id'], 'messages_received');
        
        if ($channel === 'meta') {
            $this->usageModel->incrementStat($number['user_id'], $project['id'], 'meta_messages');
        } else {
            $this->usageModel->incrementStat($number['user_id'], $project['id'], 'baileys_messages');
        }
        
        $message = $this->messageModel->find($messageId);
        
        $this->webhookService->dispatchMessageReceived($project, $message);
        
        Response::success(['status' => 'processed', 'message_id' => $message['uuid']]);
    }
    
    private function processMessageStatus(array $payload): void
    {
        $externalId = $payload['message_id'] ?? $payload['external_id'] ?? null;
        $status = $payload['status'] ?? null;
        
        if (!$externalId || !$status) {
            return;
        }
        
        $message = $this->messageModel->findByExternalId($externalId);
        if (!$message) {
            return;
        }
        
        $this->messageModel->updateStatus($message['id'], $status);
        
        $projects = $this->projectModel->getProjectsByNumber($message['whatsapp_number_id']);
        if (!empty($projects)) {
            $message = $this->messageModel->find($message['id']);
            $this->webhookService->dispatchMessageStatus($projects[0], $message, $status);
        }
        
        Response::success(['status' => 'processed']);
    }
    
    private function processConnectionUpdate(array $payload): void
    {
        $whatsappNumberId = $payload['whatsapp_number_id'] ?? null;
        $connectionStatus = $payload['status'] ?? null;
        
        if (!$whatsappNumberId) {
            return;
        }
        
        $number = $this->numberModel->find($whatsappNumberId);
        if (!$number) {
            return;
        }
        
        if ($connectionStatus === 'connected') {
            $this->numberModel->setConnected($whatsappNumberId, 'baileys');
            $this->numberModel->clearQrToken($whatsappNumberId);
        } elseif ($connectionStatus === 'disconnected') {
            $this->numberModel->setDisconnected($whatsappNumberId);
        }
        
        $projects = $this->projectModel->getProjectsByNumber($whatsappNumberId);
        foreach ($projects as $project) {
            $this->webhookService->dispatchConnectionStatus($project, $connectionStatus, $number['phone_number']);
        }
        
        Response::success(['status' => 'processed']);
    }
    
    private function processQrUpdate(array $payload): void
    {
        Response::success(['status' => 'processed']);
    }
    
    private function updateMessageStatus(array $status): void
    {
        $messageId = $status['id'] ?? null;
        $statusValue = $status['status'] ?? null;
        
        if (!$messageId || !$statusValue) {
            return;
        }
        
        $message = $this->messageModel->findByExternalId($messageId);
        if ($message) {
            $this->messageModel->updateStatus($message['id'], $statusValue);
        }
    }
    
    private function determineMessageType(array $msg, string $channel): string
    {
        if ($channel === 'meta') {
            return $msg['type'] ?? 'text';
        }
        
        if (isset($msg['imageMessage'])) return 'image';
        if (isset($msg['videoMessage'])) return 'video';
        if (isset($msg['audioMessage'])) return 'audio';
        if (isset($msg['documentMessage'])) return 'document';
        if (isset($msg['stickerMessage'])) return 'sticker';
        if (isset($msg['locationMessage'])) return 'location';
        if (isset($msg['contactMessage'])) return 'contact';
        
        return 'text';
    }
    
    private function extractContent(array $msg, string $channel): ?string
    {
        if ($channel === 'meta') {
            if (isset($msg['text']['body'])) {
                return $msg['text']['body'];
            }
            if (isset($msg['image']['caption'])) {
                return $msg['image']['caption'];
            }
            if (isset($msg['video']['caption'])) {
                return $msg['video']['caption'];
            }
            return null;
        }
        
        if (isset($msg['conversation'])) {
            return $msg['conversation'];
        }
        if (isset($msg['extendedTextMessage']['text'])) {
            return $msg['extendedTextMessage']['text'];
        }
        if (isset($msg['imageMessage']['caption'])) {
            return $msg['imageMessage']['caption'];
        }
        if (isset($msg['videoMessage']['caption'])) {
            return $msg['videoMessage']['caption'];
        }
        
        return null;
    }
    
    private function extractMediaUrl(array $msg, string $channel): ?string
    {
        if ($channel === 'meta') {
            foreach (['image', 'video', 'audio', 'document', 'sticker'] as $type) {
                if (isset($msg[$type]['id'])) {
                    return $msg[$type]['id'];
                }
            }
        }
        
        return $msg['media_url'] ?? null;
    }
    
    public function logs(): void
    {
        $user = \App\Middleware\AuthMiddleware::authenticate();
        
        $page = max(1, (int)($_GET['page'] ?? 1));
        $limit = min(100, max(10, (int)($_GET['limit'] ?? 50)));
        $offset = ($page - 1) * $limit;
        
        $webhookLogModel = new WebhookLog();
        $logs = $webhookLogModel->getUserWebhookLogs($user['id'], $limit, $offset);
        
        Response::success(['logs' => $logs]);
    }
    
    public function stats(): void
    {
        $user = \App\Middleware\AuthMiddleware::authenticate();
        
        $webhookLogModel = new WebhookLog();
        $stats = $webhookLogModel->getWebhookStats($user['id']);
        
        Response::success(['stats' => $stats]);
    }
}
