<?php

namespace App\Controllers;

use App\Models\WhatsappNumber;
use App\Models\Subscription;
use App\Models\AuditLog;
use App\Services\GatewayService;
use App\Middleware\AuthMiddleware;
use App\Utils\Response;
use App\Utils\Validator;
use App\Utils\Helpers;

class WhatsappNumberController
{
    private WhatsappNumber $numberModel;
    private Subscription $subscriptionModel;
    private AuditLog $auditLog;
    private GatewayService $gatewayService;
    
    public function __construct()
    {
        $this->numberModel = new WhatsappNumber();
        $this->subscriptionModel = new Subscription();
        $this->auditLog = new AuditLog();
        $this->gatewayService = new GatewayService();
    }
    
    public function index(): void
    {
        $user = AuthMiddleware::authenticate();
        $numbers = $this->numberModel->getActiveUserNumbers($user['id']);
        Response::success(['numbers' => $numbers]);
    }
    
    public function show(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        $number = $this->numberModel->findUserNumber($user['id'], $id);
        
        if (!$number) {
            Response::notFound('WhatsApp number not found');
        }
        
        $status = $this->gatewayService->getConnectionStatus($id);
        $number['connection_status'] = $status;
        
        Response::success(['number' => $number]);
    }
    
    public function store(): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        Validator::make($data)
            ->required('phone_number')
            ->phone('phone_number')
            ->required('channel')
            ->in('channel', ['meta', 'baileys', 'hybrid'])
            ->validate();
        
        $subscription = $this->subscriptionModel->getActiveSubscription($user['id']);
        if (!$subscription) {
            Response::error('No active subscription', 403);
        }
        
        $currentCount = $this->numberModel->countUserNumbers($user['id']);
        if ($currentCount >= $subscription['whatsapp_numbers_limit']) {
            Response::error('WhatsApp numbers limit reached for your package', 403);
        }
        
        $phoneNumber = Helpers::normalizePhoneNumber($data['phone_number']);
        if ($this->numberModel->findByPhoneNumber($phoneNumber)) {
            Response::error('This phone number is already registered', 400);
        }
        
        if ($data['channel'] === 'meta' && !$subscription['meta_enabled']) {
            Response::error('Meta Cloud API is not enabled for your package', 403);
        }
        
        if (in_array($data['channel'], ['baileys', 'hybrid']) && !$subscription['baileys_enabled']) {
            Response::error('Baileys is not enabled for your package', 403);
        }
        
        $numberId = $this->numberModel->createNumber([
            'user_id' => $user['id'],
            'phone_number' => $phoneNumber,
            'display_name' => $data['display_name'] ?? null,
            'channel' => $data['channel'],
            'status' => 'pending',
            'meta_phone_number_id' => $data['meta_phone_number_id'] ?? null,
            'meta_access_token' => $data['meta_access_token'] ?? null,
            'meta_business_account_id' => $data['meta_business_account_id'] ?? null
        ]);
        
        $this->auditLog->log($user['id'], 'whatsapp_number.created', 'whatsapp_number', $numberId);
        
        $number = $this->numberModel->find($numberId);
        Response::success(['number' => $number], 'WhatsApp number added successfully', 201);
    }
    
    public function update(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        $data = Helpers::getRequestBody();
        
        $number = $this->numberModel->findUserNumber($user['id'], $id);
        if (!$number) {
            Response::notFound('WhatsApp number not found');
        }
        
        $updateData = [];
        
        if (isset($data['display_name'])) {
            $updateData['display_name'] = $data['display_name'];
        }
        
        if (isset($data['meta_phone_number_id'])) {
            $updateData['meta_phone_number_id'] = $data['meta_phone_number_id'];
        }
        
        if (isset($data['meta_access_token'])) {
            $updateData['meta_access_token'] = $data['meta_access_token'];
        }
        
        if (isset($data['meta_business_account_id'])) {
            $updateData['meta_business_account_id'] = $data['meta_business_account_id'];
        }
        
        if (!empty($updateData)) {
            $this->numberModel->updateNumber($id, $updateData);
            $this->auditLog->log($user['id'], 'whatsapp_number.updated', 'whatsapp_number', $id);
        }
        
        $number = $this->numberModel->find($id);
        Response::success(['number' => $number], 'WhatsApp number updated');
    }
    
    public function delete(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $number = $this->numberModel->findUserNumber($user['id'], $id);
        if (!$number) {
            Response::notFound('WhatsApp number not found');
        }
        
        if ($number['status'] === 'connected') {
            $this->gatewayService->disconnectNumber($id);
        }
        
        $this->numberModel->update($id, ['is_active' => 0]);
        $this->auditLog->log($user['id'], 'whatsapp_number.deleted', 'whatsapp_number', $id);
        
        Response::success(null, 'WhatsApp number removed');
    }
    
    public function requestQr(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $number = $this->numberModel->findUserNumber($user['id'], $id);
        if (!$number) {
            Response::notFound('WhatsApp number not found');
        }
        
        if (!in_array($number['channel'], ['baileys', 'hybrid'])) {
            Response::error('QR code is only available for Baileys channel', 400);
        }
        
        $result = $this->gatewayService->requestQr($user['id'], $id);
        
        if ($result['success']) {
            $this->numberModel->setQrToken($id, $result['qr_token']);
            Response::success(['qr_token' => $result['qr_token']]);
        } else {
            Response::error('Failed to generate QR code', 500);
        }
    }
    
    public function qrStatus(): void
    {
        $token = $_GET['token'] ?? null;
        
        if (!$token) {
            Response::error('QR token required', 400);
        }
        
        $result = $this->gatewayService->getQrStatus($token);
        Response::success($result);
    }
    
    public function disconnect(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $number = $this->numberModel->findUserNumber($user['id'], $id);
        if (!$number) {
            Response::notFound('WhatsApp number not found');
        }
        
        $this->gatewayService->disconnectNumber($id);
        $this->numberModel->setDisconnected($id);
        $this->auditLog->log($user['id'], 'whatsapp_number.disconnected', 'whatsapp_number', $id);
        
        Response::success(null, 'WhatsApp disconnected');
    }
    
    public function connectionStatus(int $id): void
    {
        $user = AuthMiddleware::authenticate();
        
        $number = $this->numberModel->findUserNumber($user['id'], $id);
        if (!$number) {
            Response::notFound('WhatsApp number not found');
        }
        
        $status = $this->gatewayService->getConnectionStatus($id);
        Response::success($status);
    }
}
