<?php

namespace App\Middleware;

use App\Models\User;
use App\Models\ApiKey;
use App\Utils\Response;
use App\Utils\Helpers;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;
use App\Config\App;

class AuthMiddleware
{
    public static function authenticate(): ?array
    {
        $token = self::getBearerToken();
        
        if (!$token) {
            Response::unauthorized('No authentication token provided');
        }
        
        try {
            $decoded = JWT::decode($token, new Key(App::getJwtSecret(), 'HS256'));
            $user = (new User())->find($decoded->user_id);
            
            if (!$user) {
                Response::unauthorized('User not found');
            }
            
            if ($user['status'] !== 'active') {
                Response::forbidden('Account is ' . $user['status']);
            }
            
            return $user;
        } catch (\Exception $e) {
            Response::unauthorized('Invalid or expired token');
        }
        
        return null;
    }
    
    public static function authenticateAdmin(): ?array
    {
        $user = self::authenticate();
        
        if (!in_array($user['role'], ['admin', 'superadmin'])) {
            Response::forbidden('Admin access required');
        }
        
        return $user;
    }
    
    public static function authenticateSuperAdmin(): ?array
    {
        $user = self::authenticate();
        
        if ($user['role'] !== 'superadmin') {
            Response::forbidden('Super admin access required');
        }
        
        return $user;
    }
    
    public static function authenticateApi(): ?array
    {
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? null;
        $apiSecret = $_SERVER['HTTP_X_API_SECRET'] ?? null;
        
        if (!$apiKey || !$apiSecret) {
            Response::unauthorized('API key and secret required');
        }
        
        $apiKeyModel = new ApiKey();
        $keyData = $apiKeyModel->validateApiKey($apiKey, $apiSecret);
        
        if (!$keyData) {
            Response::unauthorized('Invalid API credentials');
        }
        
        if ($keyData['user_status'] !== 'active') {
            Response::forbidden('Account is ' . $keyData['user_status']);
        }
        
        if ($keyData['number_status'] !== 'connected') {
            Response::error('WhatsApp number is not connected', 400);
        }
        
        $ip = Helpers::getClientIp();
        if (!$apiKeyModel->checkIpWhitelist($keyData, $ip)) {
            Response::forbidden('IP address not whitelisted');
        }
        
        $apiKeyModel->updateLastUsed($keyData['id'], $ip);
        
        return $keyData;
    }
    
    public static function optionalAuth(): ?array
    {
        $token = self::getBearerToken();
        
        if (!$token) {
            return null;
        }
        
        try {
            $decoded = JWT::decode($token, new Key(App::getJwtSecret(), 'HS256'));
            return (new User())->find($decoded->user_id);
        } catch (\Exception $e) {
            return null;
        }
    }
    
    private static function getBearerToken(): ?string
    {
        $authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
        
        if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
            return $matches[1];
        }
        
        return null;
    }
    
    public static function generateToken(array $user): string
    {
        $payload = [
            'user_id' => $user['id'],
            'email' => $user['email'],
            'role' => $user['role'],
            'iat' => time(),
            'exp' => time() + App::getJwtExpiry()
        ];
        
        return JWT::encode($payload, App::getJwtSecret(), 'HS256');
    }
    
    public static function refreshToken(array $user): string
    {
        return self::generateToken($user);
    }
}
