<?php

namespace App\Middleware;

use App\Utils\Response;
use App\Utils\Helpers;

class RateLimitMiddleware
{
    private static string $cacheDir = __DIR__ . '/../../storage/cache/rate_limit/';
    
    public static function check(int $limit = 60, int $window = 60, ?string $identifier = null): void
    {
        $identifier = $identifier ?? Helpers::getClientIp();
        $key = md5($identifier);
        
        self::ensureCacheDir();
        
        $file = self::$cacheDir . $key . '.json';
        $now = time();
        
        $data = self::loadData($file);
        
        $data['requests'] = array_filter(
            $data['requests'] ?? [],
            fn($timestamp) => $timestamp > ($now - $window)
        );
        
        if (count($data['requests']) >= $limit) {
            $retryAfter = min($data['requests']) + $window - $now;
            header('X-RateLimit-Limit: ' . $limit);
            header('X-RateLimit-Remaining: 0');
            header('X-RateLimit-Reset: ' . ($now + $retryAfter));
            header('Retry-After: ' . $retryAfter);
            Response::rateLimited();
        }
        
        $data['requests'][] = $now;
        self::saveData($file, $data);
        
        header('X-RateLimit-Limit: ' . $limit);
        header('X-RateLimit-Remaining: ' . ($limit - count($data['requests'])));
        header('X-RateLimit-Reset: ' . ($now + $window));
    }
    
    public static function checkApiKey(array $keyData): void
    {
        $limit = $keyData['api_rate_limit'] ?? 60;
        $identifier = 'api_' . $keyData['api_key'];
        self::check($limit, 60, $identifier);
    }
    
    private static function ensureCacheDir(): void
    {
        if (!is_dir(self::$cacheDir)) {
            mkdir(self::$cacheDir, 0755, true);
        }
    }
    
    private static function loadData(string $file): array
    {
        if (!file_exists($file)) {
            return ['requests' => []];
        }
        
        $content = file_get_contents($file);
        return json_decode($content, true) ?? ['requests' => []];
    }
    
    private static function saveData(string $file, array $data): void
    {
        file_put_contents($file, json_encode($data));
    }
    
    public static function cleanup(): void
    {
        self::ensureCacheDir();
        $files = glob(self::$cacheDir . '*.json');
        $now = time();
        
        foreach ($files as $file) {
            if (filemtime($file) < ($now - 3600)) {
                unlink($file);
            }
        }
    }
}
