<?php

namespace App\Models;

use App\Utils\Helpers;

class ApiKey extends BaseModel
{
    protected string $table = 'api_keys';
    
    protected array $fillable = [
        'user_id', 'project_id', 'name', 'api_key', 'api_secret',
        'permissions', 'ip_whitelist', 'rate_limit', 'last_used_at',
        'last_used_ip', 'expires_at', 'is_active'
    ];
    
    protected array $hidden = ['api_secret'];
    
    public function getUserApiKeys(int $userId): array
    {
        $stmt = $this->db->prepare(
            "SELECT ak.*, p.name as project_name, wn.phone_number
             FROM {$this->table} ak
             JOIN projects p ON ak.project_id = p.id
             JOIN whatsapp_numbers wn ON p.whatsapp_number_id = wn.id
             WHERE ak.user_id = ? AND ak.is_active = 1
             ORDER BY ak.created_at DESC"
        );
        $stmt->execute([$userId]);
        return array_map(fn($row) => $this->hideFields($row), $stmt->fetchAll());
    }
    
    public function getProjectApiKeys(int $projectId): array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE project_id = ? AND is_active = 1 ORDER BY created_at DESC"
        );
        $stmt->execute([$projectId]);
        return array_map(fn($row) => $this->hideFields($row), $stmt->fetchAll());
    }
    
    public function findByApiKey(string $apiKey): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT ak.*, p.webhook_url, p.webhook_secret, p.webhook_enabled,
                    wn.id as whatsapp_number_id, wn.phone_number, wn.channel, wn.status as number_status,
                    u.id as user_id, u.email as user_email, u.status as user_status,
                    s.id as subscription_id, s.messages_used, s.status as subscription_status,
                    pkg.messages_limit, pkg.api_rate_limit, pkg.meta_enabled, pkg.baileys_enabled
             FROM {$this->table} ak
             JOIN projects p ON ak.project_id = p.id
             JOIN whatsapp_numbers wn ON p.whatsapp_number_id = wn.id
             JOIN users u ON ak.user_id = u.id
             LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
             LEFT JOIN packages pkg ON s.package_id = pkg.id
             WHERE ak.api_key = ? AND ak.is_active = 1"
        );
        $stmt->execute([$apiKey]);
        return $stmt->fetch() ?: null;
    }
    
    public function validateApiKey(string $apiKey, string $apiSecret): ?array
    {
        $keyData = $this->findByApiKey($apiKey);
        
        if (!$keyData) {
            return null;
        }
        
        $stmt = $this->db->prepare("SELECT api_secret FROM {$this->table} WHERE api_key = ?");
        $stmt->execute([$apiKey]);
        $secretData = $stmt->fetch();
        
        if (!$secretData || !hash_equals($secretData['api_secret'], $apiSecret)) {
            return null;
        }
        
        if ($keyData['expires_at'] && strtotime($keyData['expires_at']) < time()) {
            return null;
        }
        
        return $keyData;
    }
    
    public function createApiKey(int $userId, int $projectId, string $name): array
    {
        $apiKey = Helpers::generateApiKey();
        $apiSecret = Helpers::generateApiSecret();
        
        $id = $this->create([
            'user_id' => $userId,
            'project_id' => $projectId,
            'name' => $name,
            'api_key' => $apiKey,
            'api_secret' => $apiSecret,
            'is_active' => 1
        ]);
        
        return [
            'id' => $id,
            'api_key' => $apiKey,
            'api_secret' => $apiSecret
        ];
    }
    
    public function regenerateSecret(int $apiKeyId): string
    {
        $newSecret = Helpers::generateApiSecret();
        $this->update($apiKeyId, ['api_secret' => $newSecret]);
        return $newSecret;
    }
    
    public function updateLastUsed(int $apiKeyId, string $ip): bool
    {
        return $this->execute(
            "UPDATE {$this->table} SET last_used_at = NOW(), last_used_ip = ? WHERE id = ?",
            [$ip, $apiKeyId]
        );
    }
    
    public function setIpWhitelist(int $apiKeyId, array $ips): bool
    {
        return $this->update($apiKeyId, [
            'ip_whitelist' => json_encode($ips)
        ]);
    }
    
    public function checkIpWhitelist(array $keyData, string $ip): bool
    {
        if (empty($keyData['ip_whitelist'])) {
            return true;
        }
        
        $whitelist = json_decode($keyData['ip_whitelist'], true);
        if (empty($whitelist)) {
            return true;
        }
        
        return in_array($ip, $whitelist);
    }
    
    public function revokeApiKey(int $apiKeyId): bool
    {
        return $this->update($apiKeyId, ['is_active' => 0]);
    }
    
    public function findUserApiKey(int $userId, int $apiKeyId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE id = ? AND user_id = ?"
        );
        $stmt->execute([$apiKeyId, $userId]);
        return $stmt->fetch() ?: null;
    }
}
