<?php

namespace App\Models;

class DailyUsage extends BaseModel
{
    protected string $table = 'daily_usage';
    
    protected array $fillable = [
        'user_id', 'project_id', 'date', 'messages_sent', 'messages_received',
        'messages_failed', 'meta_messages', 'baileys_messages', 'api_calls',
        'webhooks_sent', 'webhooks_failed', 'cost'
    ];
    
    public function incrementStat(int $userId, ?int $projectId, string $stat, int $amount = 1): bool
    {
        $date = date('Y-m-d');
        
        $existing = $this->findUsage($userId, $projectId, $date);
        
        if ($existing) {
            return $this->execute(
                "UPDATE {$this->table} SET {$stat} = {$stat} + ? WHERE id = ?",
                [$amount, $existing['id']]
            );
        }
        
        return (bool)$this->create([
            'user_id' => $userId,
            'project_id' => $projectId,
            'date' => $date,
            $stat => $amount
        ]);
    }
    
    public function addCost(int $userId, ?int $projectId, float $cost): bool
    {
        $date = date('Y-m-d');
        
        $existing = $this->findUsage($userId, $projectId, $date);
        
        if ($existing) {
            return $this->execute(
                "UPDATE {$this->table} SET cost = cost + ? WHERE id = ?",
                [$cost, $existing['id']]
            );
        }
        
        return (bool)$this->create([
            'user_id' => $userId,
            'project_id' => $projectId,
            'date' => $date,
            'cost' => $cost
        ]);
    }
    
    private function findUsage(int $userId, ?int $projectId, string $date): ?array
    {
        if ($projectId) {
            $stmt = $this->db->prepare(
                "SELECT * FROM {$this->table} WHERE user_id = ? AND project_id = ? AND date = ?"
            );
            $stmt->execute([$userId, $projectId, $date]);
        } else {
            $stmt = $this->db->prepare(
                "SELECT * FROM {$this->table} WHERE user_id = ? AND project_id IS NULL AND date = ?"
            );
            $stmt->execute([$userId, $date]);
        }
        
        return $stmt->fetch() ?: null;
    }
    
    public function getUserUsage(int $userId, string $fromDate, string $toDate): array
    {
        $stmt = $this->db->prepare(
            "SELECT date,
                    SUM(messages_sent) as messages_sent,
                    SUM(messages_received) as messages_received,
                    SUM(messages_failed) as messages_failed,
                    SUM(meta_messages) as meta_messages,
                    SUM(baileys_messages) as baileys_messages,
                    SUM(api_calls) as api_calls,
                    SUM(webhooks_sent) as webhooks_sent,
                    SUM(webhooks_failed) as webhooks_failed,
                    SUM(cost) as cost
             FROM {$this->table}
             WHERE user_id = ? AND date BETWEEN ? AND ?
             GROUP BY date
             ORDER BY date ASC"
        );
        $stmt->execute([$userId, $fromDate, $toDate]);
        return $stmt->fetchAll();
    }
    
    public function getProjectUsage(int $projectId, string $fromDate, string $toDate): array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE project_id = ? AND date BETWEEN ? AND ? ORDER BY date ASC"
        );
        $stmt->execute([$projectId, $fromDate, $toDate]);
        return $stmt->fetchAll();
    }
    
    public function getTotalUsage(int $userId): array
    {
        $stmt = $this->db->prepare(
            "SELECT 
                SUM(messages_sent) as total_sent,
                SUM(messages_received) as total_received,
                SUM(messages_failed) as total_failed,
                SUM(api_calls) as total_api_calls,
                SUM(cost) as total_cost
             FROM {$this->table}
             WHERE user_id = ?"
        );
        $stmt->execute([$userId]);
        return $stmt->fetch() ?: [];
    }
    
    public function getSystemUsage(string $fromDate, string $toDate): array
    {
        $stmt = $this->db->prepare(
            "SELECT date,
                    SUM(messages_sent) as messages_sent,
                    SUM(messages_received) as messages_received,
                    SUM(messages_failed) as messages_failed,
                    SUM(meta_messages) as meta_messages,
                    SUM(baileys_messages) as baileys_messages,
                    SUM(cost) as cost,
                    COUNT(DISTINCT user_id) as active_users
             FROM {$this->table}
             WHERE date BETWEEN ? AND ?
             GROUP BY date
             ORDER BY date ASC"
        );
        $stmt->execute([$fromDate, $toDate]);
        return $stmt->fetchAll();
    }
}
