<?php

namespace App\Models;

use App\Utils\Helpers;

class Message extends BaseModel
{
    protected string $table = 'messages';
    
    protected array $fillable = [
        'uuid', 'user_id', 'project_id', 'whatsapp_number_id', 'api_key_id',
        'direction', 'channel', 'from_number', 'to_number', 'message_type',
        'content', 'media_url', 'media_mime_type', 'media_filename', 'media_size',
        'template_name', 'template_params', 'external_id', 'status',
        'error_code', 'error_message', 'cost', 'metadata',
        'sent_at', 'delivered_at', 'read_at'
    ];
    
    public function createMessage(array $data): int
    {
        $data['uuid'] = Helpers::generateUuid();
        
        if (isset($data['from_number'])) {
            $data['from_number'] = Helpers::normalizePhoneNumber($data['from_number']);
        }
        if (isset($data['to_number'])) {
            $data['to_number'] = Helpers::normalizePhoneNumber($data['to_number']);
        }
        
        return $this->create($data);
    }
    
    public function getUserMessages(int $userId, int $limit = 50, int $offset = 0, array $filters = []): array
    {
        $sql = "SELECT m.*, p.name as project_name, wn.phone_number as whatsapp_number
                FROM {$this->table} m
                JOIN projects p ON m.project_id = p.id
                JOIN whatsapp_numbers wn ON m.whatsapp_number_id = wn.id
                WHERE m.user_id = ?";
        $params = [$userId];
        
        if (!empty($filters['project_id'])) {
            $sql .= " AND m.project_id = ?";
            $params[] = $filters['project_id'];
        }
        
        if (!empty($filters['direction'])) {
            $sql .= " AND m.direction = ?";
            $params[] = $filters['direction'];
        }
        
        if (!empty($filters['status'])) {
            $sql .= " AND m.status = ?";
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['channel'])) {
            $sql .= " AND m.channel = ?";
            $params[] = $filters['channel'];
        }
        
        if (!empty($filters['from_date'])) {
            $sql .= " AND m.created_at >= ?";
            $params[] = $filters['from_date'];
        }
        
        if (!empty($filters['to_date'])) {
            $sql .= " AND m.created_at <= ?";
            $params[] = $filters['to_date'];
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND (m.from_number LIKE ? OR m.to_number LIKE ? OR m.content LIKE ?)";
            $search = '%' . $filters['search'] . '%';
            $params[] = $search;
            $params[] = $search;
            $params[] = $search;
        }
        
        $sql .= " ORDER BY m.created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }
    
    public function getProjectMessages(int $projectId, int $limit = 50, int $offset = 0): array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE project_id = ? ORDER BY created_at DESC LIMIT ? OFFSET ?"
        );
        $stmt->execute([$projectId, $limit, $offset]);
        return $stmt->fetchAll();
    }
    
    public function getConversation(int $projectId, string $contactNumber, int $limit = 100): array
    {
        $contactNumber = Helpers::normalizePhoneNumber($contactNumber);
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} 
             WHERE project_id = ? AND (from_number = ? OR to_number = ?)
             ORDER BY created_at ASC LIMIT ?"
        );
        $stmt->execute([$projectId, $contactNumber, $contactNumber, $limit]);
        return $stmt->fetchAll();
    }
    
    public function updateStatus(int $messageId, string $status, ?string $externalId = null): bool
    {
        $data = ['status' => $status];
        
        if ($externalId) {
            $data['external_id'] = $externalId;
        }
        
        if ($status === 'sent') {
            $data['sent_at'] = date('Y-m-d H:i:s');
        } elseif ($status === 'delivered') {
            $data['delivered_at'] = date('Y-m-d H:i:s');
        } elseif ($status === 'read') {
            $data['read_at'] = date('Y-m-d H:i:s');
        }
        
        return $this->update($messageId, $data);
    }
    
    public function setFailed(int $messageId, string $errorCode, string $errorMessage): bool
    {
        return $this->update($messageId, [
            'status' => 'failed',
            'error_code' => $errorCode,
            'error_message' => $errorMessage
        ]);
    }
    
    public function findByExternalId(string $externalId): ?array
    {
        return $this->findBy('external_id', $externalId);
    }
    
    public function findByUuid(string $uuid): ?array
    {
        return $this->findBy('uuid', $uuid);
    }
    
    public function countUserMessages(int $userId, array $filters = []): int
    {
        $sql = "SELECT COUNT(*) as count FROM {$this->table} WHERE user_id = ?";
        $params = [$userId];
        
        if (!empty($filters['direction'])) {
            $sql .= " AND direction = ?";
            $params[] = $filters['direction'];
        }
        
        if (!empty($filters['status'])) {
            $sql .= " AND status = ?";
            $params[] = $filters['status'];
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return (int)$stmt->fetch()['count'];
    }
    
    public function getMessageStats(int $userId, string $period = 'day'): array
    {
        $groupBy = match($period) {
            'hour' => "DATE_FORMAT(created_at, '%Y-%m-%d %H:00')",
            'day' => "DATE(created_at)",
            'week' => "YEARWEEK(created_at)",
            'month' => "DATE_FORMAT(created_at, '%Y-%m')",
            default => "DATE(created_at)"
        };
        
        $stmt = $this->db->prepare(
            "SELECT {$groupBy} as period,
                    COUNT(*) as total,
                    SUM(CASE WHEN direction = 'outbound' THEN 1 ELSE 0 END) as sent,
                    SUM(CASE WHEN direction = 'inbound' THEN 1 ELSE 0 END) as received,
                    SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                    SUM(cost) as total_cost
             FROM {$this->table}
             WHERE user_id = ?
             GROUP BY {$groupBy}
             ORDER BY period DESC
             LIMIT 30"
        );
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
    
    public function exportMessages(int $userId, array $filters = []): array
    {
        return $this->getUserMessages($userId, 10000, 0, $filters);
    }
}
