<?php

namespace App\Models;

use App\Utils\Helpers;

class Project extends BaseModel
{
    protected string $table = 'projects';
    
    protected array $fillable = [
        'user_id', 'whatsapp_number_id', 'name', 'description',
        'webhook_url', 'webhook_secret', 'webhook_enabled',
        'webhook_retry_count', 'webhook_retry_delay', 'is_active'
    ];
    
    public function getUserProjects(int $userId): array
    {
        $stmt = $this->db->prepare(
            "SELECT p.*, wn.phone_number, wn.display_name as number_display_name, wn.channel, wn.status as number_status
             FROM {$this->table} p
             JOIN whatsapp_numbers wn ON p.whatsapp_number_id = wn.id
             WHERE p.user_id = ? AND p.is_active = 1
             ORDER BY p.created_at DESC"
        );
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
    
    public function getProjectsByNumber(int $whatsappNumberId): array
    {
        return $this->findAllBy('whatsapp_number_id', $whatsappNumberId);
    }
    
    public function countUserProjects(int $userId): int
    {
        return $this->count('user_id = ? AND is_active = 1', [$userId]);
    }
    
    public function countProjectsForNumber(int $whatsappNumberId): int
    {
        return $this->count('whatsapp_number_id = ? AND is_active = 1', [$whatsappNumberId]);
    }
    
    public function findUserProject(int $userId, int $projectId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT p.*, wn.phone_number, wn.display_name as number_display_name, wn.channel, wn.status as number_status
             FROM {$this->table} p
             JOIN whatsapp_numbers wn ON p.whatsapp_number_id = wn.id
             WHERE p.id = ? AND p.user_id = ?"
        );
        $stmt->execute([$projectId, $userId]);
        return $stmt->fetch() ?: null;
    }
    
    public function createProject(array $data): int
    {
        if (!isset($data['webhook_secret'])) {
            $data['webhook_secret'] = Helpers::generateToken(32);
        }
        
        return $this->create($data);
    }
    
    public function updateWebhook(int $projectId, string $webhookUrl, bool $enabled = true): bool
    {
        return $this->update($projectId, [
            'webhook_url' => $webhookUrl,
            'webhook_enabled' => $enabled ? 1 : 0
        ]);
    }
    
    public function regenerateWebhookSecret(int $projectId): string
    {
        $secret = Helpers::generateToken(32);
        $this->update($projectId, ['webhook_secret' => $secret]);
        return $secret;
    }
    
    public function getProjectWithApiKeys(int $projectId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT p.*, wn.phone_number, wn.channel, wn.status as number_status,
                    u.email as user_email, u.name as user_name
             FROM {$this->table} p
             JOIN whatsapp_numbers wn ON p.whatsapp_number_id = wn.id
             JOIN users u ON p.user_id = u.id
             WHERE p.id = ?"
        );
        $stmt->execute([$projectId]);
        $project = $stmt->fetch();
        
        if (!$project) {
            return null;
        }
        
        $apiKeyModel = new ApiKey();
        $project['api_keys'] = $apiKeyModel->getProjectApiKeys($projectId);
        
        return $project;
    }
}
