<?php

namespace App\Models;

class Subscription extends BaseModel
{
    protected string $table = 'subscriptions';
    
    protected array $fillable = [
        'user_id', 'package_id', 'billing_cycle', 'status',
        'starts_at', 'ends_at', 'cancelled_at', 'trial_ends_at',
        'messages_used', 'messages_reset_at'
    ];
    
    public function getActiveSubscription(int $userId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT s.*, p.name as package_name, p.messages_limit, p.whatsapp_numbers_limit,
                    p.projects_per_number_limit, p.api_rate_limit, p.meta_enabled, p.baileys_enabled
             FROM {$this->table} s
             JOIN packages p ON s.package_id = p.id
             WHERE s.user_id = ? AND s.status = 'active'
             ORDER BY s.created_at DESC LIMIT 1"
        );
        $stmt->execute([$userId]);
        return $stmt->fetch() ?: null;
    }
    
    public function createSubscription(int $userId, int $packageId, string $billingCycle = 'monthly'): int
    {
        $this->cancelActiveSubscriptions($userId);
        
        $startsAt = date('Y-m-d H:i:s');
        $endsAt = $billingCycle === 'yearly' 
            ? date('Y-m-d H:i:s', strtotime('+1 year'))
            : date('Y-m-d H:i:s', strtotime('+1 month'));
        
        return $this->create([
            'user_id' => $userId,
            'package_id' => $packageId,
            'billing_cycle' => $billingCycle,
            'status' => 'active',
            'starts_at' => $startsAt,
            'ends_at' => $endsAt,
            'messages_used' => 0,
            'messages_reset_at' => $startsAt
        ]);
    }
    
    public function cancelActiveSubscriptions(int $userId): bool
    {
        return $this->execute(
            "UPDATE {$this->table} SET status = 'cancelled', cancelled_at = NOW() 
             WHERE user_id = ? AND status = 'active'",
            [$userId]
        );
    }
    
    public function incrementMessageUsage(int $subscriptionId, int $count = 1): bool
    {
        return $this->execute(
            "UPDATE {$this->table} SET messages_used = messages_used + ? WHERE id = ?",
            [$count, $subscriptionId]
        );
    }
    
    public function resetMessageUsage(int $subscriptionId): bool
    {
        return $this->execute(
            "UPDATE {$this->table} SET messages_used = 0, messages_reset_at = NOW() WHERE id = ?",
            [$subscriptionId]
        );
    }
    
    public function checkQuota(int $userId): array
    {
        $subscription = $this->getActiveSubscription($userId);
        
        if (!$subscription) {
            return [
                'has_quota' => false,
                'messages_used' => 0,
                'messages_limit' => 0,
                'remaining' => 0
            ];
        }
        
        $remaining = $subscription['messages_limit'] - $subscription['messages_used'];
        
        return [
            'has_quota' => $remaining > 0,
            'messages_used' => $subscription['messages_used'],
            'messages_limit' => $subscription['messages_limit'],
            'remaining' => max(0, $remaining)
        ];
    }
    
    public function getExpiredSubscriptions(): array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE status = 'active' AND ends_at < NOW()"
        );
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    public function expireSubscription(int $subscriptionId): bool
    {
        return $this->update($subscriptionId, ['status' => 'expired']);
    }
}
