<?php

namespace App\Models;

class SystemSetting extends BaseModel
{
    protected string $table = 'system_settings';
    
    protected array $fillable = [
        'key', 'value', 'type', 'description', 'is_public'
    ];
    
    private static array $cache = [];
    
    public function get(string $key, $default = null)
    {
        if (isset(self::$cache[$key])) {
            return self::$cache[$key];
        }
        
        $setting = $this->findBy('key', $key);
        
        if (!$setting) {
            return $default;
        }
        
        $value = $this->castValue($setting['value'], $setting['type']);
        self::$cache[$key] = $value;
        
        return $value;
    }
    
    public function set(string $key, $value, string $type = 'string', ?string $description = null): bool
    {
        $existing = $this->findBy('key', $key);
        
        if (is_array($value) || is_object($value)) {
            $value = json_encode($value);
            $type = 'json';
        } elseif (is_bool($value)) {
            $value = $value ? '1' : '0';
            $type = 'boolean';
        }
        
        if ($existing) {
            self::$cache[$key] = $this->castValue($value, $type);
            return $this->update($existing['id'], ['value' => $value, 'type' => $type]);
        }
        
        self::$cache[$key] = $this->castValue($value, $type);
        return (bool)$this->create([
            'key' => $key,
            'value' => $value,
            'type' => $type,
            'description' => $description
        ]);
    }
    
    public function getPublicSettings(): array
    {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE is_public = 1");
        $stmt->execute();
        $settings = $stmt->fetchAll();
        
        $result = [];
        foreach ($settings as $setting) {
            $result[$setting['key']] = $this->castValue($setting['value'], $setting['type']);
        }
        
        return $result;
    }
    
    public function getAllSettings(): array
    {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} ORDER BY `key`");
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    private function castValue($value, string $type)
    {
        return match($type) {
            'integer' => (int)$value,
            'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'json' => json_decode($value, true),
            default => $value
        };
    }
    
    public function clearCache(): void
    {
        self::$cache = [];
    }
}
