<?php

namespace App\Models;

use App\Utils\Helpers;

class User extends BaseModel
{
    protected string $table = 'users';
    
    protected array $fillable = [
        'uuid', 'email', 'password', 'name', 'phone', 'company_name',
        'avatar', 'role', 'status', 'email_verified_at', 'google_id',
        'timezone', 'language', 'two_factor_enabled', 'two_factor_secret',
        'last_login_at', 'last_login_ip'
    ];
    
    protected array $hidden = ['password', 'two_factor_secret'];
    
    public function findByEmail(string $email): ?array
    {
        $stmt = $this->db->prepare("SELECT * FROM {$this->table} WHERE email = ?");
        $stmt->execute([$email]);
        return $stmt->fetch() ?: null;
    }
    
    public function findByGoogleId(string $googleId): ?array
    {
        return $this->findBy('google_id', $googleId);
    }
    
    public function createUser(array $data): int
    {
        $data['uuid'] = Helpers::generateUuid();
        
        if (isset($data['password'])) {
            $data['password'] = Helpers::hashPassword($data['password']);
        }
        
        return $this->create($data);
    }
    
    public function updatePassword(int $userId, string $password): bool
    {
        return $this->update($userId, [
            'password' => Helpers::hashPassword($password)
        ]);
    }
    
    public function verifyEmail(int $userId): bool
    {
        return $this->execute(
            "UPDATE {$this->table} SET email_verified_at = NOW() WHERE id = ?",
            [$userId]
        );
    }
    
    public function updateLastLogin(int $userId, string $ip): bool
    {
        return $this->execute(
            "UPDATE {$this->table} SET last_login_at = NOW(), last_login_ip = ? WHERE id = ?",
            [$ip, $userId]
        );
    }
    
    public function getActiveClients(int $limit = 100, int $offset = 0): array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE role = 'client' AND status = 'active' 
             ORDER BY created_at DESC LIMIT ? OFFSET ?"
        );
        $stmt->execute([$limit, $offset]);
        return array_map(fn($row) => $this->hideFields($row), $stmt->fetchAll());
    }
    
    public function getAllUsers(int $limit = 100, int $offset = 0, ?string $role = null, ?string $status = null): array
    {
        $sql = "SELECT * FROM {$this->table} WHERE 1=1";
        $params = [];
        
        if ($role) {
            $sql .= " AND role = ?";
            $params[] = $role;
        }
        
        if ($status) {
            $sql .= " AND status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return array_map(fn($row) => $this->hideFields($row), $stmt->fetchAll());
    }
    
    public function countUsers(?string $role = null, ?string $status = null): int
    {
        $sql = "SELECT COUNT(*) as count FROM {$this->table} WHERE 1=1";
        $params = [];
        
        if ($role) {
            $sql .= " AND role = ?";
            $params[] = $role;
        }
        
        if ($status) {
            $sql .= " AND status = ?";
            $params[] = $status;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return (int)$stmt->fetch()['count'];
    }
    
    public function suspendUser(int $userId): bool
    {
        return $this->update($userId, ['status' => 'suspended']);
    }
    
    public function banUser(int $userId): bool
    {
        return $this->update($userId, ['status' => 'banned']);
    }
    
    public function activateUser(int $userId): bool
    {
        return $this->update($userId, ['status' => 'active']);
    }
    
    public function getUserWithSubscription(int $userId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT u.*, s.id as subscription_id, s.package_id, s.status as subscription_status,
                    s.messages_used, s.ends_at as subscription_ends_at,
                    p.name as package_name, p.messages_limit, p.whatsapp_numbers_limit,
                    p.projects_per_number_limit, p.api_rate_limit, p.meta_enabled, p.baileys_enabled
             FROM {$this->table} u
             LEFT JOIN subscriptions s ON u.id = s.user_id AND s.status = 'active'
             LEFT JOIN packages p ON s.package_id = p.id
             WHERE u.id = ?"
        );
        $stmt->execute([$userId]);
        $result = $stmt->fetch();
        return $result ? $this->hideFields($result) : null;
    }
}
