<?php

namespace App\Models;

class WebhookLog extends BaseModel
{
    protected string $table = 'webhook_logs';
    
    protected array $fillable = [
        'user_id', 'project_id', 'message_id', 'webhook_url', 'event_type',
        'payload', 'response_status', 'response_body', 'attempt', 'max_attempts',
        'status', 'next_retry_at', 'completed_at', 'duration_ms'
    ];
    
    public function createLog(array $data): int
    {
        if (isset($data['payload']) && is_array($data['payload'])) {
            $data['payload'] = json_encode($data['payload']);
        }
        
        return $this->create($data);
    }
    
    public function getUserWebhookLogs(int $userId, int $limit = 50, int $offset = 0): array
    {
        $stmt = $this->db->prepare(
            "SELECT wl.*, p.name as project_name
             FROM {$this->table} wl
             JOIN projects p ON wl.project_id = p.id
             WHERE wl.user_id = ?
             ORDER BY wl.created_at DESC
             LIMIT ? OFFSET ?"
        );
        $stmt->execute([$userId, $limit, $offset]);
        return $stmt->fetchAll();
    }
    
    public function getPendingRetries(): array
    {
        $stmt = $this->db->prepare(
            "SELECT wl.*, p.webhook_url, p.webhook_secret
             FROM {$this->table} wl
             JOIN projects p ON wl.project_id = p.id
             WHERE wl.status = 'retrying' AND wl.next_retry_at <= NOW() AND wl.attempt < wl.max_attempts"
        );
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    public function markSuccess(int $logId, int $responseStatus, string $responseBody, int $durationMs): bool
    {
        return $this->update($logId, [
            'status' => 'success',
            'response_status' => $responseStatus,
            'response_body' => $responseBody,
            'duration_ms' => $durationMs,
            'completed_at' => date('Y-m-d H:i:s')
        ]);
    }
    
    public function markFailed(int $logId, int $responseStatus, string $responseBody, int $durationMs): bool
    {
        $log = $this->find($logId);
        
        if ($log && $log['attempt'] < $log['max_attempts']) {
            $nextRetry = date('Y-m-d H:i:s', strtotime('+' . (60 * pow(2, $log['attempt'])) . ' seconds'));
            return $this->update($logId, [
                'status' => 'retrying',
                'response_status' => $responseStatus,
                'response_body' => $responseBody,
                'duration_ms' => $durationMs,
                'attempt' => $log['attempt'] + 1,
                'next_retry_at' => $nextRetry
            ]);
        }
        
        return $this->update($logId, [
            'status' => 'failed',
            'response_status' => $responseStatus,
            'response_body' => $responseBody,
            'duration_ms' => $durationMs,
            'completed_at' => date('Y-m-d H:i:s')
        ]);
    }
    
    public function getWebhookStats(int $userId): array
    {
        $stmt = $this->db->prepare(
            "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) as successful,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
                SUM(CASE WHEN status = 'retrying' THEN 1 ELSE 0 END) as pending,
                AVG(duration_ms) as avg_duration
             FROM {$this->table}
             WHERE user_id = ?"
        );
        $stmt->execute([$userId]);
        return $stmt->fetch() ?: [];
    }
}
