<?php

namespace App\Models;

use App\Utils\Helpers;

class WhatsappNumber extends BaseModel
{
    protected string $table = 'whatsapp_numbers';
    
    protected array $fillable = [
        'user_id', 'phone_number', 'display_name', 'channel', 'status',
        'meta_phone_number_id', 'meta_access_token', 'meta_business_account_id',
        'meta_webhook_verified', 'baileys_session_id', 'baileys_qr_token',
        'baileys_connected_at', 'last_active_at', 'is_active'
    ];
    
    protected array $hidden = ['meta_access_token'];
    
    public function getUserNumbers(int $userId): array
    {
        return $this->findAllBy('user_id', $userId);
    }
    
    public function getActiveUserNumbers(int $userId): array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE user_id = ? AND is_active = 1 ORDER BY created_at DESC"
        );
        $stmt->execute([$userId]);
        return array_map(fn($row) => $this->hideFields($row), $stmt->fetchAll());
    }
    
    public function countUserNumbers(int $userId): int
    {
        return $this->count('user_id = ? AND is_active = 1', [$userId]);
    }
    
    public function findByPhoneNumber(string $phoneNumber): ?array
    {
        $phoneNumber = Helpers::normalizePhoneNumber($phoneNumber);
        return $this->findBy('phone_number', $phoneNumber);
    }
    
    public function findUserNumber(int $userId, int $numberId): ?array
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM {$this->table} WHERE id = ? AND user_id = ?"
        );
        $stmt->execute([$numberId, $userId]);
        $result = $stmt->fetch();
        return $result ? $this->hideFields($result) : null;
    }
    
    public function createNumber(array $data): int
    {
        if (isset($data['phone_number'])) {
            $data['phone_number'] = Helpers::normalizePhoneNumber($data['phone_number']);
        }
        
        if (isset($data['meta_access_token'])) {
            $data['meta_access_token'] = Helpers::encryptToken($data['meta_access_token']);
        }
        
        return $this->create($data);
    }
    
    public function updateNumber(int $id, array $data): bool
    {
        if (isset($data['meta_access_token'])) {
            $data['meta_access_token'] = Helpers::encryptToken($data['meta_access_token']);
        }
        
        return $this->update($id, $data);
    }
    
    public function setConnected(int $numberId, string $channel): bool
    {
        $data = ['status' => 'connected', 'last_active_at' => date('Y-m-d H:i:s')];
        
        if ($channel === 'baileys') {
            $data['baileys_connected_at'] = date('Y-m-d H:i:s');
        }
        
        return $this->update($numberId, $data);
    }
    
    public function setDisconnected(int $numberId): bool
    {
        return $this->update($numberId, ['status' => 'disconnected']);
    }
    
    public function setQrToken(int $numberId, string $token): bool
    {
        return $this->update($numberId, [
            'baileys_qr_token' => $token,
            'status' => 'pending'
        ]);
    }
    
    public function findByQrToken(string $token): ?array
    {
        return $this->findBy('baileys_qr_token', $token);
    }
    
    public function clearQrToken(int $numberId): bool
    {
        return $this->update($numberId, ['baileys_qr_token' => null]);
    }
    
    public function getDecryptedAccessToken(int $numberId): ?string
    {
        $stmt = $this->db->prepare("SELECT meta_access_token FROM {$this->table} WHERE id = ?");
        $stmt->execute([$numberId]);
        $result = $stmt->fetch();
        
        if ($result && $result['meta_access_token']) {
            return Helpers::decryptToken($result['meta_access_token']);
        }
        
        return null;
    }
    
    public function getConnectedNumbers(): array
    {
        $stmt = $this->db->prepare(
            "SELECT wn.*, u.email as user_email, u.name as user_name
             FROM {$this->table} wn
             JOIN users u ON wn.user_id = u.id
             WHERE wn.status = 'connected' AND wn.is_active = 1"
        );
        $stmt->execute();
        return $stmt->fetchAll();
    }
    
    public function updateLastActive(int $numberId): bool
    {
        return $this->execute(
            "UPDATE {$this->table} SET last_active_at = NOW() WHERE id = ?",
            [$numberId]
        );
    }
}
