<?php

namespace App\Services;

use App\Models\User;
use App\Models\Package;
use App\Models\Subscription;
use App\Models\AuditLog;
use App\Models\Credits;
use App\Middleware\AuthMiddleware;
use App\Utils\Helpers;
use App\Utils\Response;
use App\Config\App;
use Google\Client as GoogleClient;

class AuthService
{
    private User $userModel;
    private AuditLog $auditLog;
    
    public function __construct()
    {
        $this->userModel = new User();
        $this->auditLog = new AuditLog();
    }
    
    public function register(array $data): array
    {
        if ($this->userModel->exists('email', $data['email'])) {
            Response::error('Email already registered', 400);
        }
        
        $userId = $this->userModel->createUser([
            'email' => $data['email'],
            'password' => $data['password'],
            'name' => $data['name'],
            'phone' => $data['phone'] ?? null,
            'company_name' => $data['company_name'] ?? null,
            'role' => 'client',
            'status' => 'active'
        ]);
        
        $this->assignDefaultPackage($userId);
        $this->initializeCredits($userId);
        
        $user = $this->userModel->find($userId);
        $token = AuthMiddleware::generateToken($user);
        
        $this->auditLog->log($userId, 'user.registered', 'user', $userId);
        
        return [
            'user' => $user,
            'token' => $token,
            'token_type' => 'Bearer',
            'expires_in' => App::getJwtExpiry()
        ];
    }
    
    public function login(string $email, string $password): array
    {
        $user = $this->userModel->findByEmail($email);
        
        if (!$user) {
            Response::error('Invalid credentials', 401);
        }
        
        if (!$user['password'] || !Helpers::verifyPassword($password, $user['password'])) {
            Response::error('Invalid credentials', 401);
        }
        
        if ($user['status'] !== 'active') {
            Response::error('Account is ' . $user['status'], 403);
        }
        
        $this->userModel->updateLastLogin($user['id'], Helpers::getClientIp());
        
        $token = AuthMiddleware::generateToken($user);
        
        $this->auditLog->log($user['id'], 'user.login', 'user', $user['id']);
        
        unset($user['password'], $user['two_factor_secret']);
        
        return [
            'user' => $user,
            'token' => $token,
            'token_type' => 'Bearer',
            'expires_in' => App::getJwtExpiry()
        ];
    }
    
    public function googleAuth(string $code): array
    {
        $client = new GoogleClient();
        $client->setClientId($_ENV['GOOGLE_CLIENT_ID']);
        $client->setClientSecret($_ENV['GOOGLE_CLIENT_SECRET']);
        $client->setRedirectUri($_ENV['GOOGLE_REDIRECT_URI']);
        
        try {
            $token = $client->fetchAccessTokenWithAuthCode($code);
            
            if (isset($token['error'])) {
                Response::error('Google authentication failed: ' . $token['error'], 400);
            }
            
            $client->setAccessToken($token);
            
            $oauth2 = new \Google\Service\Oauth2($client);
            $googleUser = $oauth2->userinfo->get();
            
            $email = $googleUser->email;
            $googleId = $googleUser->id;
            $name = $googleUser->name;
            $avatar = $googleUser->picture;
            
        } catch (\Exception $e) {
            Response::error('Google authentication failed: ' . $e->getMessage(), 400);
        }
        
        $user = $this->userModel->findByEmail($email);
        
        if ($user) {
            if ($user['status'] !== 'active') {
                Response::error('Account is ' . $user['status'], 403);
            }
            
            $this->userModel->update($user['id'], [
                'google_id' => $googleId,
                'avatar' => $avatar,
                'email_verified_at' => $user['email_verified_at'] ?? date('Y-m-d H:i:s')
            ]);
            
            $this->userModel->updateLastLogin($user['id'], Helpers::getClientIp());
            $this->auditLog->log($user['id'], 'user.google_login', 'user', $user['id']);
            
        } else {
            $userId = $this->userModel->createUser([
                'email' => $email,
                'name' => $name,
                'google_id' => $googleId,
                'avatar' => $avatar,
                'role' => 'client',
                'status' => 'active',
                'email_verified_at' => date('Y-m-d H:i:s')
            ]);
            
            $this->assignDefaultPackage($userId);
            $this->initializeCredits($userId);
            
            $user = $this->userModel->find($userId);
            $this->auditLog->log($userId, 'user.google_registered', 'user', $userId);
        }
        
        $user = $this->userModel->find($user['id']);
        $token = AuthMiddleware::generateToken($user);
        
        unset($user['password'], $user['two_factor_secret']);
        
        return [
            'user' => $user,
            'token' => $token,
            'token_type' => 'Bearer',
            'expires_in' => App::getJwtExpiry()
        ];
    }
    
    public function getGoogleAuthUrl(): string
    {
        $client = new GoogleClient();
        $client->setClientId($_ENV['GOOGLE_CLIENT_ID']);
        $client->setClientSecret($_ENV['GOOGLE_CLIENT_SECRET']);
        $client->setRedirectUri($_ENV['GOOGLE_REDIRECT_URI']);
        $client->addScope('email');
        $client->addScope('profile');
        $client->setAccessType('offline');
        $client->setPrompt('select_account');
        
        return $client->createAuthUrl();
    }
    
    public function logout(int $userId): void
    {
        $this->auditLog->log($userId, 'user.logout', 'user', $userId);
    }
    
    public function refreshToken(array $user): array
    {
        $token = AuthMiddleware::generateToken($user);
        
        return [
            'token' => $token,
            'token_type' => 'Bearer',
            'expires_in' => App::getJwtExpiry()
        ];
    }
    
    private function assignDefaultPackage(int $userId): void
    {
        $packageModel = new Package();
        $defaultPackage = $packageModel->getDefaultPackage();
        
        if ($defaultPackage) {
            $subscriptionModel = new Subscription();
            $subscriptionModel->createSubscription($userId, $defaultPackage['id']);
        }
    }
    
    private function initializeCredits(int $userId): void
    {
        $db = \App\Config\Database::getInstance();
        $stmt = $db->prepare("INSERT INTO credits (user_id, balance) VALUES (?, 0) ON DUPLICATE KEY UPDATE user_id = user_id");
        $stmt->execute([$userId]);
    }
}
