<?php

namespace App\Services;

use App\Config\App;
use App\Models\Message;
use App\Models\Subscription;
use App\Models\DailyUsage;
use App\Models\RouteSetting;
use App\Utils\Response;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

class GatewayService
{
    private Client $httpClient;
    private Message $messageModel;
    private Subscription $subscriptionModel;
    private DailyUsage $usageModel;
    private RouteSetting $routeSettingModel;
    
    public function __construct()
    {
        $this->httpClient = new Client([
            'base_uri' => App::getGatewayUrl(),
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Gateway-Secret' => App::getGatewaySecret()
            ]
        ]);
        
        $this->messageModel = new Message();
        $this->subscriptionModel = new Subscription();
        $this->usageModel = new DailyUsage();
        $this->routeSettingModel = new RouteSetting();
    }
    
    public function sendMessage(array $apiKeyData, array $messageData): array
    {
        $this->checkQuota($apiKeyData);
        $channel = $this->determineChannel($apiKeyData);
        
        $messageId = $this->messageModel->createMessage([
            'user_id' => $apiKeyData['user_id'],
            'project_id' => $apiKeyData['project_id'],
            'whatsapp_number_id' => $apiKeyData['whatsapp_number_id'],
            'api_key_id' => $apiKeyData['id'],
            'direction' => 'outbound',
            'channel' => $channel,
            'from_number' => $apiKeyData['phone_number'],
            'to_number' => $messageData['to'],
            'message_type' => $messageData['type'] ?? 'text',
            'content' => $messageData['message'] ?? null,
            'status' => 'pending'
        ]);
        
        try {
            $response = $this->httpClient->post('/send', [
                'json' => [
                    'message_id' => $messageId,
                    'channel' => $channel,
                    'from' => $apiKeyData['phone_number'],
                    'to' => $messageData['to'],
                    'type' => $messageData['type'] ?? 'text',
                    'content' => $messageData['message'] ?? null,
                    'whatsapp_number_id' => $apiKeyData['whatsapp_number_id'],
                    'meta_phone_number_id' => $apiKeyData['meta_phone_number_id'] ?? null
                ]
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['success']) {
                $this->messageModel->updateStatus($messageId, 'sent', $result['external_id'] ?? null);
                $this->recordUsage($apiKeyData, $channel);
                
                $message = $this->messageModel->find($messageId);
                
                return [
                    'success' => true,
                    'message_id' => $message['uuid'],
                    'channel' => $channel
                ];
            } else {
                $this->messageModel->setFailed($messageId, $result['error_code'] ?? 'UNKNOWN', $result['error'] ?? 'Unknown error');
                Response::error($result['error'] ?? 'Failed to send message', 400);
            }
            
        } catch (GuzzleException $e) {
            $this->messageModel->setFailed($messageId, 'GATEWAY_ERROR', $e->getMessage());
            Response::error('Gateway service unavailable', 503);
        }
        
        return ['success' => false];
    }
    
    public function sendMedia(array $apiKeyData, array $mediaData): array
    {
        $this->checkQuota($apiKeyData);
        $channel = $this->determineChannel($apiKeyData);
        
        $messageId = $this->messageModel->createMessage([
            'user_id' => $apiKeyData['user_id'],
            'project_id' => $apiKeyData['project_id'],
            'whatsapp_number_id' => $apiKeyData['whatsapp_number_id'],
            'api_key_id' => $apiKeyData['id'],
            'direction' => 'outbound',
            'channel' => $channel,
            'from_number' => $apiKeyData['phone_number'],
            'to_number' => $mediaData['to'],
            'message_type' => $mediaData['type'] ?? 'image',
            'content' => $mediaData['caption'] ?? null,
            'media_url' => $mediaData['media_url'],
            'status' => 'pending'
        ]);
        
        try {
            $response = $this->httpClient->post('/send-media', [
                'json' => [
                    'message_id' => $messageId,
                    'channel' => $channel,
                    'from' => $apiKeyData['phone_number'],
                    'to' => $mediaData['to'],
                    'type' => $mediaData['type'] ?? 'image',
                    'media_url' => $mediaData['media_url'],
                    'caption' => $mediaData['caption'] ?? null,
                    'filename' => $mediaData['filename'] ?? null,
                    'whatsapp_number_id' => $apiKeyData['whatsapp_number_id']
                ]
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['success']) {
                $this->messageModel->updateStatus($messageId, 'sent', $result['external_id'] ?? null);
                $this->recordUsage($apiKeyData, $channel);
                
                $message = $this->messageModel->find($messageId);
                
                return [
                    'success' => true,
                    'message_id' => $message['uuid'],
                    'channel' => $channel
                ];
            } else {
                $this->messageModel->setFailed($messageId, $result['error_code'] ?? 'UNKNOWN', $result['error'] ?? 'Unknown error');
                Response::error($result['error'] ?? 'Failed to send media', 400);
            }
            
        } catch (GuzzleException $e) {
            $this->messageModel->setFailed($messageId, 'GATEWAY_ERROR', $e->getMessage());
            Response::error('Gateway service unavailable', 503);
        }
        
        return ['success' => false];
    }
    
    public function requestQr(int $userId, int $whatsappNumberId): array
    {
        try {
            $response = $this->httpClient->post('/request-qr', [
                'json' => [
                    'user_id' => $userId,
                    'whatsapp_number_id' => $whatsappNumberId
                ]
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            return [
                'success' => $result['success'] ?? false,
                'qr_token' => $result['qr_token'] ?? null
            ];
            
        } catch (GuzzleException $e) {
            Response::error('Gateway service unavailable', 503);
        }
        
        return ['success' => false];
    }
    
    public function getQrStatus(string $token): array
    {
        try {
            $response = $this->httpClient->get('/qr-status', [
                'query' => ['token' => $token]
            ]);
            
            return json_decode($response->getBody()->getContents(), true);
            
        } catch (GuzzleException $e) {
            return ['success' => false, 'status' => 'error'];
        }
    }
    
    public function disconnectNumber(int $whatsappNumberId): bool
    {
        try {
            $response = $this->httpClient->post('/disconnect', [
                'json' => ['whatsapp_number_id' => $whatsappNumberId]
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            return $result['success'] ?? false;
            
        } catch (GuzzleException $e) {
            return false;
        }
    }
    
    public function getConnectionStatus(int $whatsappNumberId): array
    {
        try {
            $response = $this->httpClient->get('/connection-status', [
                'query' => ['whatsapp_number_id' => $whatsappNumberId]
            ]);
            
            return json_decode($response->getBody()->getContents(), true);
            
        } catch (GuzzleException $e) {
            return ['connected' => false, 'status' => 'unknown'];
        }
    }
    
    private function checkQuota(array $apiKeyData): void
    {
        if (!$apiKeyData['subscription_id']) {
            Response::error('No active subscription', 403);
        }
        
        $remaining = $apiKeyData['messages_limit'] - $apiKeyData['messages_used'];
        
        if ($remaining <= 0) {
            Response::quotaExceeded();
        }
    }
    
    private function determineChannel(array $apiKeyData): string
    {
        $numberChannel = $apiKeyData['channel'] ?? 'baileys';
        
        if ($numberChannel === 'meta' && $apiKeyData['meta_enabled']) {
            if ($this->routeSettingModel->isRouteEnabled('meta')) {
                return 'meta';
            }
        }
        
        if ($apiKeyData['baileys_enabled'] && $this->routeSettingModel->isRouteEnabled('baileys')) {
            return 'baileys';
        }
        
        if ($this->routeSettingModel->isRouteEnabled('meta') && $apiKeyData['meta_enabled']) {
            return 'meta';
        }
        
        Response::error('No available messaging channel', 400);
        return 'baileys';
    }
    
    private function recordUsage(array $apiKeyData, string $channel): void
    {
        $this->subscriptionModel->incrementMessageUsage($apiKeyData['subscription_id']);
        
        $this->usageModel->incrementStat($apiKeyData['user_id'], $apiKeyData['project_id'], 'messages_sent');
        
        if ($channel === 'meta') {
            $this->usageModel->incrementStat($apiKeyData['user_id'], $apiKeyData['project_id'], 'meta_messages');
        } else {
            $this->usageModel->incrementStat($apiKeyData['user_id'], $apiKeyData['project_id'], 'baileys_messages');
        }
        
        $cost = $this->routeSettingModel->getRouteCost($channel);
        if ($cost > 0) {
            $this->usageModel->addCost($apiKeyData['user_id'], $apiKeyData['project_id'], $cost);
        }
    }
}
