<?php

namespace App\Services;

use App\Models\Project;
use App\Models\Message;
use App\Models\WebhookLog;
use App\Models\DailyUsage;
use App\Utils\Helpers;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

class WebhookService
{
    private WebhookLog $webhookLogModel;
    private DailyUsage $usageModel;
    private Client $httpClient;
    
    public function __construct()
    {
        $this->webhookLogModel = new WebhookLog();
        $this->usageModel = new DailyUsage();
        $this->httpClient = new Client([
            'timeout' => 30,
            'http_errors' => false
        ]);
    }
    
    public function dispatchMessageReceived(array $project, array $message): void
    {
        if (!$project['webhook_enabled'] || empty($project['webhook_url'])) {
            return;
        }
        
        $payload = [
            'event' => 'message.received',
            'client_id' => $project['user_id'],
            'project_id' => $project['id'],
            'from' => $message['from_number'],
            'to' => $message['to_number'],
            'message' => $message['content'],
            'message_type' => $message['message_type'],
            'message_id' => $message['uuid'],
            'timestamp' => strtotime($message['created_at']),
            'channel' => $message['channel'],
            'media_url' => $message['media_url'] ?? null
        ];
        
        $this->dispatch($project, $payload, 'message.received', $message['id']);
    }
    
    public function dispatchMessageStatus(array $project, array $message, string $status): void
    {
        if (!$project['webhook_enabled'] || empty($project['webhook_url'])) {
            return;
        }
        
        $payload = [
            'event' => 'message.status',
            'client_id' => $project['user_id'],
            'project_id' => $project['id'],
            'message_id' => $message['uuid'],
            'external_id' => $message['external_id'],
            'status' => $status,
            'timestamp' => time(),
            'channel' => $message['channel']
        ];
        
        $this->dispatch($project, $payload, 'message.status', $message['id']);
    }
    
    public function dispatchConnectionStatus(array $project, string $status, string $phoneNumber): void
    {
        if (!$project['webhook_enabled'] || empty($project['webhook_url'])) {
            return;
        }
        
        $payload = [
            'event' => 'connection.status',
            'client_id' => $project['user_id'],
            'project_id' => $project['id'],
            'phone_number' => $phoneNumber,
            'status' => $status,
            'timestamp' => time()
        ];
        
        $this->dispatch($project, $payload, 'connection.status');
    }
    
    private function dispatch(array $project, array $payload, string $eventType, ?int $messageId = null): void
    {
        $logId = $this->webhookLogModel->createLog([
            'user_id' => $project['user_id'],
            'project_id' => $project['id'],
            'message_id' => $messageId,
            'webhook_url' => $project['webhook_url'],
            'event_type' => $eventType,
            'payload' => $payload,
            'max_attempts' => $project['webhook_retry_count'] ?? 3,
            'status' => 'pending'
        ]);
        
        $this->sendWebhook($logId, $project['webhook_url'], $project['webhook_secret'], $payload);
    }
    
    public function sendWebhook(int $logId, string $url, string $secret, array $payload): bool
    {
        $payloadJson = json_encode($payload);
        $signature = Helpers::generateHmacSignature($payloadJson, $secret);
        
        $startTime = microtime(true);
        
        try {
            $response = $this->httpClient->post($url, [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'X-Signature' => $signature,
                    'X-Webhook-Event' => $payload['event'] ?? 'unknown',
                    'User-Agent' => 'WhatsAppGateway/1.0'
                ],
                'body' => $payloadJson
            ]);
            
            $durationMs = (int)((microtime(true) - $startTime) * 1000);
            $statusCode = $response->getStatusCode();
            $responseBody = $response->getBody()->getContents();
            
            if ($statusCode >= 200 && $statusCode < 300) {
                $this->webhookLogModel->markSuccess($logId, $statusCode, $responseBody, $durationMs);
                $this->usageModel->incrementStat($payload['client_id'] ?? 0, $payload['project_id'] ?? null, 'webhooks_sent');
                return true;
            } else {
                $this->webhookLogModel->markFailed($logId, $statusCode, $responseBody, $durationMs);
                $this->usageModel->incrementStat($payload['client_id'] ?? 0, $payload['project_id'] ?? null, 'webhooks_failed');
                return false;
            }
            
        } catch (GuzzleException $e) {
            $durationMs = (int)((microtime(true) - $startTime) * 1000);
            $this->webhookLogModel->markFailed($logId, 0, $e->getMessage(), $durationMs);
            $this->usageModel->incrementStat($payload['client_id'] ?? 0, $payload['project_id'] ?? null, 'webhooks_failed');
            return false;
        }
    }
    
    public function retryPendingWebhooks(): int
    {
        $pending = $this->webhookLogModel->getPendingRetries();
        $retried = 0;
        
        foreach ($pending as $log) {
            $payload = json_decode($log['payload'], true);
            if ($this->sendWebhook($log['id'], $log['webhook_url'], $log['webhook_secret'], $payload)) {
                $retried++;
            }
        }
        
        return $retried;
    }
}
