<?php

namespace App\Utils;

use Ramsey\Uuid\Uuid;

class Helpers
{
    public static function generateUuid(): string
    {
        return Uuid::uuid4()->toString();
    }
    
    public static function generateApiKey(): string
    {
        return 'wag_' . bin2hex(random_bytes(24));
    }
    
    public static function generateApiSecret(): string
    {
        return bin2hex(random_bytes(32));
    }
    
    public static function generateToken(int $length = 32): string
    {
        return bin2hex(random_bytes($length));
    }
    
    public static function hashPassword(string $password): string
    {
        return password_hash($password, PASSWORD_BCRYPT, ['cost' => 10]);
    }
    
    public static function verifyPassword(string $password, string $hash): bool
    {
        return password_verify($password, $hash);
    }
    
    public static function encryptToken(string $token): string
    {
        $key = $_ENV['APP_KEY'] ?? 'default-key';
        $iv = random_bytes(16);
        $encrypted = openssl_encrypt($token, 'AES-256-CBC', $key, 0, $iv);
        return base64_encode($iv . $encrypted);
    }
    
    public static function decryptToken(string $encryptedToken): ?string
    {
        $key = $_ENV['APP_KEY'] ?? 'default-key';
        $data = base64_decode($encryptedToken);
        $iv = substr($data, 0, 16);
        $encrypted = substr($data, 16);
        $decrypted = openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
        return $decrypted !== false ? $decrypted : null;
    }
    
    public static function generateHmacSignature(string $payload, string $secret): string
    {
        return hash_hmac('sha256', $payload, $secret);
    }
    
    public static function verifyHmacSignature(string $payload, string $signature, string $secret): bool
    {
        $expected = self::generateHmacSignature($payload, $secret);
        return hash_equals($expected, $signature);
    }
    
    public static function normalizePhoneNumber(string $phone): string
    {
        $phone = preg_replace('/[^0-9+]/', '', $phone);
        if (strpos($phone, '+') !== 0) {
            $phone = '+' . $phone;
        }
        return $phone;
    }
    
    public static function getClientIp(): string
    {
        $headers = [
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return '0.0.0.0';
    }
    
    public static function getUserAgent(): string
    {
        return $_SERVER['HTTP_USER_AGENT'] ?? '';
    }
    
    public static function sanitizeString(string $string): string
    {
        return htmlspecialchars(strip_tags(trim($string)), ENT_QUOTES, 'UTF-8');
    }
    
    public static function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, $precision) . ' ' . $units[$pow];
    }
    
    public static function jsonDecode(string $json): ?array
    {
        $data = json_decode($json, true);
        return json_last_error() === JSON_ERROR_NONE ? $data : null;
    }
    
    public static function jsonEncode($data): string
    {
        return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }
    
    public static function getRequestBody(): array
    {
        $input = file_get_contents('php://input');
        return self::jsonDecode($input) ?? [];
    }
    
    public static function slug(string $string): string
    {
        $string = preg_replace('/[^a-zA-Z0-9\s]/', '', $string);
        $string = strtolower(trim($string));
        return preg_replace('/\s+/', '-', $string);
    }
}
