<?php

namespace App\Utils;

class Validator
{
    private array $errors = [];
    private array $data;
    
    public function __construct(array $data)
    {
        $this->data = $data;
    }
    
    public static function make(array $data): self
    {
        return new self($data);
    }
    
    public function required(string $field, string $message = null): self
    {
        if (!isset($this->data[$field]) || trim($this->data[$field]) === '') {
            $this->errors[$field] = $message ?? "{$field} is required";
        }
        return $this;
    }
    
    public function email(string $field, string $message = null): self
    {
        if (isset($this->data[$field]) && !filter_var($this->data[$field], FILTER_VALIDATE_EMAIL)) {
            $this->errors[$field] = $message ?? "{$field} must be a valid email";
        }
        return $this;
    }
    
    public function minLength(string $field, int $length, string $message = null): self
    {
        if (isset($this->data[$field]) && strlen($this->data[$field]) < $length) {
            $this->errors[$field] = $message ?? "{$field} must be at least {$length} characters";
        }
        return $this;
    }
    
    public function maxLength(string $field, int $length, string $message = null): self
    {
        if (isset($this->data[$field]) && strlen($this->data[$field]) > $length) {
            $this->errors[$field] = $message ?? "{$field} must not exceed {$length} characters";
        }
        return $this;
    }
    
    public function phone(string $field, string $message = null): self
    {
        if (isset($this->data[$field])) {
            $phone = preg_replace('/[^0-9+]/', '', $this->data[$field]);
            if (!preg_match('/^\+?[1-9]\d{6,14}$/', $phone)) {
                $this->errors[$field] = $message ?? "{$field} must be a valid phone number";
            }
        }
        return $this;
    }
    
    public function url(string $field, string $message = null): self
    {
        if (isset($this->data[$field]) && !filter_var($this->data[$field], FILTER_VALIDATE_URL)) {
            $this->errors[$field] = $message ?? "{$field} must be a valid URL";
        }
        return $this;
    }
    
    public function in(string $field, array $values, string $message = null): self
    {
        if (isset($this->data[$field]) && !in_array($this->data[$field], $values)) {
            $this->errors[$field] = $message ?? "{$field} must be one of: " . implode(', ', $values);
        }
        return $this;
    }
    
    public function numeric(string $field, string $message = null): self
    {
        if (isset($this->data[$field]) && !is_numeric($this->data[$field])) {
            $this->errors[$field] = $message ?? "{$field} must be numeric";
        }
        return $this;
    }
    
    public function integer(string $field, string $message = null): self
    {
        if (isset($this->data[$field]) && !filter_var($this->data[$field], FILTER_VALIDATE_INT)) {
            $this->errors[$field] = $message ?? "{$field} must be an integer";
        }
        return $this;
    }
    
    public function boolean(string $field, string $message = null): self
    {
        if (isset($this->data[$field]) && !is_bool($this->data[$field]) && !in_array($this->data[$field], [0, 1, '0', '1', true, false], true)) {
            $this->errors[$field] = $message ?? "{$field} must be a boolean";
        }
        return $this;
    }
    
    public function confirmed(string $field, string $confirmField = null, string $message = null): self
    {
        $confirmField = $confirmField ?? $field . '_confirmation';
        if (isset($this->data[$field]) && (!isset($this->data[$confirmField]) || $this->data[$field] !== $this->data[$confirmField])) {
            $this->errors[$field] = $message ?? "{$field} confirmation does not match";
        }
        return $this;
    }
    
    public function fails(): bool
    {
        return !empty($this->errors);
    }
    
    public function passes(): bool
    {
        return empty($this->errors);
    }
    
    public function errors(): array
    {
        return $this->errors;
    }
    
    public function validate(): void
    {
        if ($this->fails()) {
            Response::validationError($this->errors);
        }
    }
    
    public function getData(): array
    {
        return $this->data;
    }
    
    public function get(string $field, $default = null)
    {
        return $this->data[$field] ?? $default;
    }
}
