<?php

use App\Controllers\AuthController;
use App\Controllers\WhatsappNumberController;
use App\Controllers\ProjectController;
use App\Controllers\ApiKeyController;
use App\Controllers\MessageController;
use App\Controllers\BillingController;
use App\Controllers\WebhookController;
use App\Controllers\Admin\UserController as AdminUserController;
use App\Controllers\Admin\PackageController as AdminPackageController;
use App\Controllers\Admin\SettingsController as AdminSettingsController;
use App\Controllers\Admin\AnalyticsController as AdminAnalyticsController;

class Router
{
    private array $routes = [];
    private string $prefix = '';
    
    public function group(string $prefix, callable $callback): void
    {
        $previousPrefix = $this->prefix;
        $this->prefix .= $prefix;
        $callback($this);
        $this->prefix = $previousPrefix;
    }
    
    public function get(string $path, array $handler): void
    {
        $this->addRoute('GET', $path, $handler);
    }
    
    public function post(string $path, array $handler): void
    {
        $this->addRoute('POST', $path, $handler);
    }
    
    public function put(string $path, array $handler): void
    {
        $this->addRoute('PUT', $path, $handler);
    }
    
    public function patch(string $path, array $handler): void
    {
        $this->addRoute('PATCH', $path, $handler);
    }
    
    public function delete(string $path, array $handler): void
    {
        $this->addRoute('DELETE', $path, $handler);
    }
    
    private function addRoute(string $method, string $path, array $handler): void
    {
        $fullPath = $this->prefix . $path;
        $this->routes[] = [
            'method' => $method,
            'path' => $fullPath,
            'handler' => $handler
        ];
    }
    
    public function dispatch(string $method, string $uri): void
    {
        $uri = parse_url($uri, PHP_URL_PATH);
        $uri = rtrim($uri, '/');
        
        if (empty($uri)) {
            $uri = '/';
        }
        
        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) {
                continue;
            }
            
            $pattern = $this->convertToRegex($route['path']);
            
            if (preg_match($pattern, $uri, $matches)) {
                array_shift($matches);
                
                [$controllerClass, $action] = $route['handler'];
                $controller = new $controllerClass();
                
                call_user_func_array([$controller, $action], $matches);
                return;
            }
        }
        
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Route not found']);
    }
    
    private function convertToRegex(string $path): string
    {
        $pattern = preg_replace('/\{([a-zA-Z_]+)\}/', '([^/]+)', $path);
        return '#^' . $pattern . '$#';
    }
}

$router = new Router();

$router->group('/api', function($router) {
    
    $router->group('/auth', function($router) {
        $router->post('/register', [AuthController::class, 'register']);
        $router->post('/login', [AuthController::class, 'login']);
        $router->get('/google', [AuthController::class, 'googleAuthUrl']);
        $router->get('/google/callback', [AuthController::class, 'googleCallback']);
        $router->post('/logout', [AuthController::class, 'logout']);
        $router->get('/me', [AuthController::class, 'me']);
        $router->post('/refresh', [AuthController::class, 'refreshToken']);
        $router->put('/profile', [AuthController::class, 'updateProfile']);
        $router->post('/change-password', [AuthController::class, 'changePassword']);
    });
    
    $router->group('/whatsapp-numbers', function($router) {
        $router->get('', [WhatsappNumberController::class, 'index']);
        $router->post('', [WhatsappNumberController::class, 'store']);
        $router->get('/{id}', [WhatsappNumberController::class, 'show']);
        $router->put('/{id}', [WhatsappNumberController::class, 'update']);
        $router->delete('/{id}', [WhatsappNumberController::class, 'delete']);
        $router->post('/{id}/request-qr', [WhatsappNumberController::class, 'requestQr']);
        $router->get('/qr-status', [WhatsappNumberController::class, 'qrStatus']);
        $router->post('/{id}/disconnect', [WhatsappNumberController::class, 'disconnect']);
        $router->get('/{id}/status', [WhatsappNumberController::class, 'connectionStatus']);
    });
    
    $router->group('/projects', function($router) {
        $router->get('', [ProjectController::class, 'index']);
        $router->post('', [ProjectController::class, 'store']);
        $router->get('/{id}', [ProjectController::class, 'show']);
        $router->put('/{id}', [ProjectController::class, 'update']);
        $router->delete('/{id}', [ProjectController::class, 'delete']);
        $router->post('/{id}/regenerate-webhook-secret', [ProjectController::class, 'regenerateWebhookSecret']);
        $router->get('/{id}/with-api-keys', [ProjectController::class, 'getWithApiKeys']);
    });
    
    $router->group('/api-keys', function($router) {
        $router->get('', [ApiKeyController::class, 'index']);
        $router->post('', [ApiKeyController::class, 'store']);
        $router->get('/{id}', [ApiKeyController::class, 'show']);
        $router->put('/{id}', [ApiKeyController::class, 'update']);
        $router->delete('/{id}', [ApiKeyController::class, 'delete']);
        $router->post('/{id}/regenerate-secret', [ApiKeyController::class, 'regenerateSecret']);
        $router->post('/{id}/ip-whitelist', [ApiKeyController::class, 'setIpWhitelist']);
    });
    
    $router->post('/send-message', [MessageController::class, 'sendMessage']);
    $router->post('/send-media', [MessageController::class, 'sendMedia']);
    $router->post('/request-qr', [MessageController::class, 'requestQr']);
    $router->get('/qr-status', [MessageController::class, 'qrStatus']);
    
    $router->group('/messages', function($router) {
        $router->get('', [MessageController::class, 'index']);
        $router->get('/stats', [MessageController::class, 'stats']);
        $router->get('/export', [MessageController::class, 'export']);
        $router->get('/{id}', [MessageController::class, 'show']);
        $router->get('/conversation/{projectId}', [MessageController::class, 'conversation']);
    });
    
    $router->group('/billing', function($router) {
        $router->get('/packages', [BillingController::class, 'packages']);
        $router->get('/subscription', [BillingController::class, 'currentSubscription']);
        $router->post('/subscribe', [BillingController::class, 'subscribe']);
        $router->post('/cancel', [BillingController::class, 'cancelSubscription']);
        $router->get('/usage', [BillingController::class, 'usage']);
        $router->get('/quota', [BillingController::class, 'quota']);
        $router->get('/credits', [BillingController::class, 'credits']);
        $router->get('/invoices', [BillingController::class, 'invoices']);
        $router->get('/invoices/{id}', [BillingController::class, 'invoiceDetails']);
    });
    
    $router->group('/webhooks', function($router) {
        $router->get('/logs', [WebhookController::class, 'logs']);
        $router->get('/stats', [WebhookController::class, 'stats']);
    });
    
    $router->group('/admin', function($router) {
        
        $router->group('/users', function($router) {
            $router->get('', [AdminUserController::class, 'index']);
            $router->post('', [AdminUserController::class, 'store']);
            $router->get('/{id}', [AdminUserController::class, 'show']);
            $router->put('/{id}', [AdminUserController::class, 'update']);
            $router->post('/{id}/suspend', [AdminUserController::class, 'suspend']);
            $router->post('/{id}/ban', [AdminUserController::class, 'ban']);
            $router->post('/{id}/activate', [AdminUserController::class, 'activate']);
            $router->post('/{id}/reset-password', [AdminUserController::class, 'resetPassword']);
            $router->post('/{userId}/whatsapp/{numberId}/disconnect', [AdminUserController::class, 'disconnectWhatsapp']);
        });
        
        $router->group('/packages', function($router) {
            $router->get('', [AdminPackageController::class, 'index']);
            $router->post('', [AdminPackageController::class, 'store']);
            $router->get('/{id}', [AdminPackageController::class, 'show']);
            $router->put('/{id}', [AdminPackageController::class, 'update']);
            $router->delete('/{id}', [AdminPackageController::class, 'delete']);
            $router->post('/{id}/set-default', [AdminPackageController::class, 'setDefault']);
        });
        
        $router->group('/settings', function($router) {
            $router->get('', [AdminSettingsController::class, 'index']);
            $router->put('', [AdminSettingsController::class, 'update']);
            $router->get('/routes', [AdminSettingsController::class, 'routes']);
            $router->put('/routes', [AdminSettingsController::class, 'updateRoute']);
            $router->post('/routes/{routeName}/enable', [AdminSettingsController::class, 'enableRoute']);
            $router->post('/routes/{routeName}/disable', [AdminSettingsController::class, 'disableRoute']);
        });
        
        $router->group('/analytics', function($router) {
            $router->get('/dashboard', [AdminAnalyticsController::class, 'dashboard']);
            $router->get('/usage', [AdminAnalyticsController::class, 'usage']);
            $router->get('/revenue', [AdminAnalyticsController::class, 'revenue']);
            $router->get('/logs', [AdminAnalyticsController::class, 'logs']);
            $router->get('/top-users', [AdminAnalyticsController::class, 'topUsers']);
            $router->get('/messages', [AdminAnalyticsController::class, 'messageStats']);
        });
    });
});

$router->post('/webhooks/meta', [WebhookController::class, 'handleMeta']);
$router->get('/webhooks/meta', [WebhookController::class, 'handleMeta']);
$router->post('/webhooks/baileys', [WebhookController::class, 'handleBaileys']);

return $router;
