# WhatsApp Gateway API Documentation

## Overview

The WhatsApp Gateway API provides a unified interface for sending and receiving WhatsApp messages through both Meta Cloud API and Baileys (headless) channels.

## Base URL

```
http://your-domain.com/api
```

## Authentication

All API requests require authentication using API keys.

### Headers

```
X-API-KEY: your_api_key
X-API-SECRET: your_api_secret
```

### Getting API Keys

1. Log in to the dashboard
2. Navigate to **Projects** and create a project
3. Go to **API Keys** and generate a new key
4. Save your API secret immediately (it won't be shown again)

---

## Messaging Endpoints

### Send Text Message

Send a text message to a WhatsApp number.

**Endpoint:** `POST /api/send-message`

**Request:**
```json
{
  "to": "+923124356789",
  "message": "Hello, this is a test message",
  "type": "text"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Success",
  "data": {
    "success": true,
    "message_id": "550e8400-e29b-41d4-a716-446655440000",
    "channel": "baileys"
  }
}
```

### Send Media Message

Send an image, video, audio, or document.

**Endpoint:** `POST /api/send-media`

**Request:**
```json
{
  "to": "+923124356789",
  "media_url": "https://example.com/image.jpg",
  "caption": "Check out this image!",
  "type": "image"
}
```

**Supported Types:**
- `image` - JPEG, PNG, WebP
- `video` - MP4
- `audio` - MP3, OGG
- `document` - PDF, DOC, etc.

**Response:**
```json
{
  "success": true,
  "data": {
    "success": true,
    "message_id": "550e8400-e29b-41d4-a716-446655440001",
    "channel": "baileys"
  }
}
```

### Request QR Code (Baileys)

Request a QR code for WhatsApp Web authentication.

**Endpoint:** `POST /api/request-qr`

**Response:**
```json
{
  "success": true,
  "data": {
    "success": true,
    "qr_token": "abc123def456"
  }
}
```

### Check QR Status

Poll for QR code status.

**Endpoint:** `GET /api/qr-status?token=abc123def456`

**Response:**
```json
{
  "success": true,
  "data": {
    "success": true,
    "status": "pending",
    "qr": "data:image/png;base64,..."
  }
}
```

**Status Values:**
- `pending` - QR code ready, waiting for scan
- `connected` - Successfully connected
- `expired` - QR code expired, request new one

---

## Webhook Events

Configure your webhook URL in the project settings to receive incoming messages and status updates.

### Webhook Payload Format

All webhooks include an HMAC-SHA256 signature in the `X-Signature` header for verification.

### Message Received

```json
{
  "event": "message.received",
  "client_id": 123,
  "project_id": 456,
  "from": "+923124356789",
  "to": "+923001234567",
  "message": "Hello!",
  "message_type": "text",
  "message_id": "550e8400-e29b-41d4-a716-446655440000",
  "timestamp": 1736882450,
  "channel": "baileys",
  "media_url": null
}
```

### Message Status Update

```json
{
  "event": "message.status",
  "client_id": 123,
  "project_id": 456,
  "message_id": "550e8400-e29b-41d4-a716-446655440000",
  "external_id": "wamid.xxx",
  "status": "delivered",
  "timestamp": 1736882460,
  "channel": "baileys"
}
```

### Connection Status

```json
{
  "event": "connection.status",
  "client_id": 123,
  "project_id": 456,
  "phone_number": "+923001234567",
  "status": "connected",
  "timestamp": 1736882470
}
```

### Verifying Webhook Signatures

```php
<?php
$payload = file_get_contents('php://input');
$signature = $_SERVER['HTTP_X_SIGNATURE'];
$secret = 'your_webhook_secret';

$expected = hash_hmac('sha256', $payload, $secret);

if (hash_equals($expected, $signature)) {
    // Signature valid
    $data = json_decode($payload, true);
    // Process webhook
} else {
    // Invalid signature
    http_response_code(401);
}
```

---

## Error Responses

### Standard Error Format

```json
{
  "success": false,
  "error": "Error message here"
}
```

### Common Error Codes

| Status | Error | Description |
|--------|-------|-------------|
| 400 | Bad Request | Invalid request parameters |
| 401 | Unauthorized | Invalid or missing API credentials |
| 403 | Forbidden | Account suspended or quota exceeded |
| 404 | Not Found | Resource not found |
| 429 | Rate Limited | Too many requests |
| 500 | Server Error | Internal server error |

### Quota Exceeded

```json
{
  "success": false,
  "error": "Quota exceeded"
}
```

### Rate Limited

```json
{
  "success": false,
  "error": "Rate limit exceeded. Please try again later."
}
```

---

## Rate Limits

Rate limits are based on your subscription package:

| Package | Requests/Minute |
|---------|-----------------|
| Free | 30 |
| Starter | 60 |
| Business | 120 |
| Enterprise | 300 |

Rate limit headers are included in all responses:

```
X-RateLimit-Limit: 60
X-RateLimit-Remaining: 45
X-RateLimit-Reset: 1736882500
```

---

## Code Examples

### PHP

```php
<?php
$apiKey = 'your_api_key';
$apiSecret = 'your_api_secret';

$ch = curl_init('http://your-domain.com/api/send-message');
curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json',
        'X-API-KEY: ' . $apiKey,
        'X-API-SECRET: ' . $apiSecret
    ],
    CURLOPT_POSTFIELDS => json_encode([
        'to' => '+923124356789',
        'message' => 'Hello from PHP!',
        'type' => 'text'
    ])
]);

$response = curl_exec($ch);
$result = json_decode($response, true);

if ($result['success']) {
    echo "Message sent! ID: " . $result['data']['message_id'];
} else {
    echo "Error: " . $result['error'];
}
```

### JavaScript/Node.js

```javascript
const axios = require('axios');

const apiKey = 'your_api_key';
const apiSecret = 'your_api_secret';

async function sendMessage(to, message) {
    try {
        const response = await axios.post(
            'http://your-domain.com/api/send-message',
            { to, message, type: 'text' },
            {
                headers: {
                    'X-API-KEY': apiKey,
                    'X-API-SECRET': apiSecret
                }
            }
        );
        
        console.log('Message sent:', response.data.data.message_id);
        return response.data;
    } catch (error) {
        console.error('Error:', error.response?.data?.error);
        throw error;
    }
}

sendMessage('+923124356789', 'Hello from Node.js!');
```

### Python

```python
import requests

api_key = 'your_api_key'
api_secret = 'your_api_secret'

response = requests.post(
    'http://your-domain.com/api/send-message',
    json={
        'to': '+923124356789',
        'message': 'Hello from Python!',
        'type': 'text'
    },
    headers={
        'X-API-KEY': api_key,
        'X-API-SECRET': api_secret
    }
)

result = response.json()
if result['success']:
    print(f"Message sent! ID: {result['data']['message_id']}")
else:
    print(f"Error: {result['error']}")
```

### cURL

```bash
curl -X POST http://your-domain.com/api/send-message \
  -H "Content-Type: application/json" \
  -H "X-API-KEY: your_api_key" \
  -H "X-API-SECRET: your_api_secret" \
  -d '{"to": "+923124356789", "message": "Hello from cURL!", "type": "text"}'
```

---

## Best Practices

1. **Store credentials securely** - Never hardcode API keys in client-side code
2. **Implement webhook verification** - Always verify webhook signatures
3. **Handle rate limits** - Implement exponential backoff for retries
4. **Use idempotency** - Track message IDs to avoid duplicates
5. **Monitor quota** - Check remaining quota before sending bulk messages
