# Client Guide

## Getting Started

Welcome to the WhatsApp Gateway! This guide will help you set up and start sending WhatsApp messages through our API.

---

## Quick Start

### Step 1: Create an Account

1. Visit the registration page
2. Enter your name, email, and password
3. Optionally sign up with Google
4. You'll receive a free tier subscription automatically

### Step 2: Add a WhatsApp Number

1. Go to **WhatsApp Numbers** in the sidebar
2. Click **Add Number**
3. Enter your phone number (with country code, e.g., +923001234567)
4. Choose a channel:
   - **Baileys** - Headless WhatsApp Web (recommended for testing)
   - **Meta** - Official API (requires Meta Business setup)
   - **Hybrid** - Use both channels
5. Click **Add Number**

### Step 3: Connect via QR Code (Baileys)

1. Click **Connect** on your number
2. A QR code will appear
3. Open WhatsApp on your phone
4. Go to **Settings > Linked Devices > Link a Device**
5. Scan the QR code
6. Wait for "Connected" status

### Step 4: Create a Project

1. Go to **Projects** in the sidebar
2. Click **Create Project**
3. Enter a project name
4. Select your WhatsApp number
5. (Optional) Add a webhook URL for receiving messages
6. Click **Create Project**

### Step 5: Generate API Keys

1. Go to **API Keys** in the sidebar
2. Click **Generate API Key**
3. Enter a name and select your project
4. **Important:** Save your API Secret immediately!

### Step 6: Send Your First Message

```bash
curl -X POST https://your-gateway.com/api/send-message \
  -H "Content-Type: application/json" \
  -H "X-API-KEY: your_api_key" \
  -H "X-API-SECRET: your_api_secret" \
  -d '{"to": "+923124356789", "message": "Hello World!", "type": "text"}'
```

---

## Dashboard Overview

### Statistics Cards

- **Messages Sent** - Outbound messages today
- **Messages Received** - Inbound messages today
- **Connected Numbers** - Active WhatsApp connections
- **Quota Remaining** - Messages left in your plan

### Recent Messages

Shows your latest 5 messages with status indicators.

### Quick Actions

- Add WhatsApp Number
- Create Project
- Generate API Key

---

## Managing WhatsApp Numbers

### Adding a Number

You can add multiple WhatsApp numbers based on your subscription plan.

**Channel Options:**

| Channel | Description | Best For |
|---------|-------------|----------|
| Baileys | Uses WhatsApp Web protocol | Testing, low volume |
| Meta | Official Business API | Production, high volume |
| Hybrid | Can use either | Flexibility |

### Connection Status

| Status | Meaning |
|--------|---------|
| 🟢 Connected | Ready to send/receive |
| 🟡 Pending | Waiting for QR scan |
| 🔴 Disconnected | Needs reconnection |

### Reconnecting

If your number disconnects:
1. Click **Connect** on the number
2. Scan the new QR code
3. Wait for connection

**Common Disconnect Reasons:**
- Phone went offline for too long
- WhatsApp app was updated
- Logged out from phone
- Session expired

---

## Managing Projects

### What is a Project?

A project links your WhatsApp number to your application. Each project has:
- Unique webhook URL
- Webhook secret for verification
- Associated API keys

### Webhook Configuration

**Webhook URL:** The endpoint where we'll send incoming messages and status updates.

**Webhook Secret:** Used to verify webhook authenticity. Keep this secret!

### Regenerating Webhook Secret

If your secret is compromised:
1. Open project details
2. Click **Regenerate**
3. Update your application with the new secret

---

## Managing API Keys

### Creating Keys

Each API key is tied to a specific project. Create separate keys for:
- Development environment
- Staging environment
- Production environment

### Security Best Practices

1. **Never share** your API secret
2. **Rotate keys** periodically
3. **Use IP whitelisting** for production
4. **Revoke** compromised keys immediately

### IP Whitelisting

Restrict API access to specific IP addresses:
1. Open API key details
2. Add allowed IP addresses
3. Save changes

---

## Sending Messages

### Text Messages

```json
POST /api/send-message
{
  "to": "+923124356789",
  "message": "Hello!",
  "type": "text"
}
```

### Media Messages

```json
POST /api/send-media
{
  "to": "+923124356789",
  "media_url": "https://example.com/image.jpg",
  "caption": "Check this out!",
  "type": "image"
}
```

**Supported Media Types:**
- `image` - JPG, PNG, WebP (max 5MB)
- `video` - MP4 (max 16MB)
- `audio` - MP3, OGG (max 16MB)
- `document` - PDF, DOC, etc. (max 100MB)

---

## Receiving Messages (Webhooks)

### Setting Up Webhooks

1. Create a public endpoint on your server
2. Add the URL to your project settings
3. Enable webhooks

### Webhook Payload

```json
{
  "event": "message.received",
  "from": "+923124356789",
  "to": "+923001234567",
  "message": "Hello!",
  "message_type": "text",
  "message_id": "uuid-here",
  "timestamp": 1736882450
}
```

### Verifying Webhooks

Always verify the `X-Signature` header:

```php
$payload = file_get_contents('php://input');
$signature = $_SERVER['HTTP_X_SIGNATURE'];
$expected = hash_hmac('sha256', $payload, $webhookSecret);

if (!hash_equals($expected, $signature)) {
    http_response_code(401);
    exit;
}
```

### Responding to Webhooks

Return `200 OK` quickly to acknowledge receipt. Process messages asynchronously.

---

## Billing & Subscriptions

### Viewing Your Plan

Go to **Billing** to see:
- Current subscription
- Usage this month
- Quota remaining
- Credit balance

### Upgrading

1. Go to **Billing**
2. View available plans
3. Select a plan
4. Complete payment

### Usage Tracking

Monitor your usage to avoid hitting limits:
- Messages are counted per send
- Failed messages don't count
- Inbound messages may count depending on plan

---

## Settings

### Profile Settings

Update your:
- Name
- Phone number
- Company name
- Timezone
- Language

### Changing Password

1. Go to **Settings**
2. Enter current password
3. Enter new password (min 8 characters)
4. Confirm new password
5. Click **Change Password**

---

## Troubleshooting

### Message Not Sending

1. Check WhatsApp number is connected
2. Verify API credentials are correct
3. Ensure quota is not exceeded
4. Check recipient number format

### Webhook Not Receiving

1. Verify webhook URL is publicly accessible
2. Check webhook is enabled in project
3. Verify your server returns 200 OK
4. Check webhook logs in dashboard

### QR Code Not Appearing

1. Refresh the page
2. Check Gateway service is running
3. Try a different browser
4. Contact support if persists

### Rate Limit Errors

If you see "Rate limit exceeded":
1. Reduce request frequency
2. Implement request queuing
3. Upgrade to higher plan

---

## Best Practices

### Message Formatting

- Keep messages concise
- Use proper formatting (bold, italic)
- Include clear call-to-action
- Respect user preferences

### Compliance

- Get consent before messaging
- Provide opt-out options
- Follow WhatsApp Business Policy
- Don't send spam

### Performance

- Use webhooks instead of polling
- Implement retry logic with backoff
- Cache API responses when possible
- Monitor your usage regularly

---

## Support

### Self-Service

- Check this documentation
- Review API docs
- Check system status

### Contact Support

For issues not covered here:
- Email: support@your-domain.com
- Response time: 24-48 hours

### Feature Requests

We welcome feedback! Submit feature requests through the dashboard.

---

## FAQ

**Q: Can I use multiple WhatsApp numbers?**
A: Yes, based on your subscription plan limits.

**Q: What happens if I exceed my quota?**
A: Messages will fail with "Quota exceeded" error. Upgrade your plan or wait for reset.

**Q: How often does quota reset?**
A: Monthly, on your subscription anniversary date.

**Q: Is my data secure?**
A: Yes, we use encryption and follow security best practices.

**Q: Can I export my message history?**
A: Yes, go to Messages and click Export CSV.

**Q: What's the difference between Meta and Baileys?**
A: Meta is the official API (more reliable, costs more). Baileys uses WhatsApp Web protocol (free, less stable).
