# Deployment Guide

## System Requirements

### Server Requirements

- **OS**: Windows Server / Linux (Ubuntu 20.04+)
- **PHP**: 8.0 or higher
- **Node.js**: 18.0 or higher
- **MySQL**: 8.0 or higher
- **RAM**: Minimum 2GB (4GB recommended)
- **Storage**: 20GB minimum

### PHP Extensions

- PDO MySQL
- OpenSSL
- cURL
- JSON
- Mbstring

---

## Installation Steps

### 1. Clone/Upload Project

```bash
# Upload project files to your server
# For XAMPP on Windows:
C:\xampp\htdocs\whatsapp_api\
```

### 2. Database Setup

```bash
# Create database
mysql -u root -p
CREATE DATABASE whatsapp_gateway CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
exit;

# Import schema
mysql -u root -p whatsapp_gateway < database/schema.sql
```

### 3. PHP Backend Setup

```bash
cd backend

# Install dependencies
composer install

# Copy environment file
copy .env.example .env

# Edit .env with your settings
notepad .env
```

**Configure `.env`:**
```env
APP_NAME="WhatsApp Gateway"
APP_ENV=production
APP_DEBUG=false
APP_URL=http://your-domain.com
APP_KEY=generate-a-32-character-random-string

DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=whatsapp_gateway
DB_USERNAME=your_db_user
DB_PASSWORD=your_db_password

JWT_SECRET=generate-another-random-string
JWT_EXPIRY=86400

GATEWAY_URL=http://localhost:3000
GATEWAY_SECRET=your-gateway-secret

GOOGLE_CLIENT_ID=your-google-client-id
GOOGLE_CLIENT_SECRET=your-google-client-secret
GOOGLE_REDIRECT_URI=http://your-domain.com/api/auth/google/callback
```

### 4. Node.js Gateway Setup

```bash
cd gateway

# Install dependencies
npm install

# Copy environment file
copy .env.example .env

# Edit .env
notepad .env
```

**Configure `.env`:**
```env
PORT=3000
NODE_ENV=production
GATEWAY_SECRET=your-gateway-secret

BACKEND_URL=http://localhost/whatsapp_api/backend/public

DB_HOST=localhost
DB_PORT=3306
DB_NAME=whatsapp_gateway
DB_USER=your_db_user
DB_PASSWORD=your_db_password

BAILEYS_SESSIONS_PATH=./sessions
```

### 5. Frontend Setup

```bash
cd frontend

# Install dependencies
npm install

# Create .env
echo VITE_API_URL=http://your-domain.com/api > .env

# Build for production
npm run build
```

### 6. Web Server Configuration

#### Apache (XAMPP)

The `.htaccess` file is already configured. Ensure `mod_rewrite` is enabled.

**httpd.conf:**
```apache
LoadModule rewrite_module modules/mod_rewrite.so
```

#### Nginx (Alternative)

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /var/www/whatsapp_api/backend/public;

    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location /frontend {
        alias /var/www/whatsapp_api/frontend/dist;
        try_files $uri $uri/ /frontend/index.html;
    }
}
```

---

## Running the Services

### Start Gateway Service

**Development:**
```bash
cd gateway
npm run dev
```

**Production with PM2:**
```bash
# Install PM2 globally
npm install -g pm2

# Start gateway
cd gateway
pm2 start ecosystem.config.js

# Save PM2 process list
pm2 save

# Setup PM2 to start on boot (Linux)
pm2 startup
```

### PM2 Commands

```bash
pm2 status              # Check status
pm2 logs whatsapp-gateway  # View logs
pm2 restart whatsapp-gateway  # Restart
pm2 stop whatsapp-gateway     # Stop
```

---

## Security Configuration

### 1. Environment Variables

Never commit `.env` files to version control.

### 2. File Permissions (Linux)

```bash
chmod 755 backend/
chmod 644 backend/.env
chmod 755 gateway/
chmod 644 gateway/.env
chmod -R 777 gateway/sessions/
chmod -R 777 backend/storage/
```

### 3. Firewall Rules

```bash
# Allow HTTP
ufw allow 80/tcp

# Allow Gateway (internal only)
ufw allow from 127.0.0.1 to any port 3000
```

### 4. Rate Limiting

Rate limiting is built into the application. Configure limits in packages.

---

## Cron Jobs

### Webhook Retry (Every 5 minutes)

```bash
*/5 * * * * php /path/to/backend/cron/retry_webhooks.php
```

### Subscription Expiry Check (Daily)

```bash
0 0 * * * php /path/to/backend/cron/check_subscriptions.php
```

### Clean Old Logs (Weekly)

```bash
0 0 * * 0 php /path/to/backend/cron/cleanup_logs.php
```

---

## Monitoring

### Health Checks

**Gateway Health:**
```bash
curl http://localhost:3000/health
```

**Expected Response:**
```json
{
  "status": "healthy",
  "uptime": 12345.67,
  "timestamp": "2024-01-15T12:00:00.000Z"
}
```

### Log Files

- **Gateway Logs:** `gateway/logs/`
- **PHP Logs:** `backend/logs/`
- **PM2 Logs:** `~/.pm2/logs/`

---

## Troubleshooting

### Common Issues

#### 1. Database Connection Failed

```
Error: SQLSTATE[HY000] [2002] Connection refused
```

**Solution:** Check MySQL is running and credentials are correct.

#### 2. Gateway Not Starting

```
Error: EADDRINUSE: address already in use :::3000
```

**Solution:** Kill existing process or change port.

```bash
netstat -ano | findstr :3000
taskkill /PID <pid> /F
```

#### 3. Baileys Session Issues

```
Error: Session not found
```

**Solution:** Clear sessions folder and reconnect.

```bash
rm -rf gateway/sessions/*
```

#### 4. CORS Errors

**Solution:** Add your frontend domain to `CorsMiddleware.php`:

```php
$allowedOrigins = [
    'http://your-frontend-domain.com',
    // ...
];
```

---

## Backup Strategy

### Database Backup

```bash
# Daily backup
mysqldump -u root -p whatsapp_gateway > backup_$(date +%Y%m%d).sql

# Restore
mysql -u root -p whatsapp_gateway < backup_20240115.sql
```

### Session Backup

```bash
# Backup Baileys sessions
tar -czf sessions_backup.tar.gz gateway/sessions/
```

---

## Scaling Considerations

### Horizontal Scaling

1. Use a load balancer for PHP backend
2. Keep Gateway service on single instance (Baileys sessions are local)
3. Use Redis for session management (future enhancement)

### Database Optimization

1. Add indexes for frequently queried columns
2. Partition messages table by date
3. Archive old messages periodically

---

## Updates

### Updating the Application

```bash
# Backup first
mysqldump -u root -p whatsapp_gateway > backup_before_update.sql

# Pull updates
git pull origin main

# Update dependencies
cd backend && composer install
cd ../gateway && npm install
cd ../frontend && npm install && npm run build

# Run migrations if any
php backend/migrate.php

# Restart services
pm2 restart whatsapp-gateway
```
