import axios from 'axios'

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost/windsurf/CascadeProjects/whatsapp_api/backend/public/api'

const client = axios.create({
  baseURL: API_URL,
  headers: {
    'Content-Type': 'application/json'
  }
})

client.interceptors.request.use(
  (config) => {
    const token = localStorage.getItem('token')
    if (token) {
      config.headers.Authorization = `Bearer ${token}`
    }
    return config
  },
  (error) => Promise.reject(error)
)

client.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response?.status === 401) {
      localStorage.removeItem('token')
      localStorage.removeItem('user')
      window.location.href = '/login'
    }
    return Promise.reject(error)
  }
)

export default client

export const auth = {
  login: (data) => client.post('/auth/login', data),
  register: (data) => client.post('/auth/register', data),
  logout: () => client.post('/auth/logout'),
  me: () => client.get('/auth/me'),
  updateProfile: (data) => client.put('/auth/profile', data),
  changePassword: (data) => client.post('/auth/change-password', data),
  getGoogleAuthUrl: () => client.get('/auth/google'),
  googleCallback: (code) => client.get(`/auth/google/callback?code=${code}`)
}

export const whatsappNumbers = {
  list: () => client.get('/whatsapp-numbers'),
  get: (id) => client.get(`/whatsapp-numbers/${id}`),
  create: (data) => client.post('/whatsapp-numbers', data),
  update: (id, data) => client.put(`/whatsapp-numbers/${id}`, data),
  delete: (id) => client.delete(`/whatsapp-numbers/${id}`),
  requestQr: (id) => client.post(`/whatsapp-numbers/${id}/request-qr`),
  qrStatus: (token) => client.get(`/whatsapp-numbers/qr-status?token=${token}`),
  disconnect: (id) => client.post(`/whatsapp-numbers/${id}/disconnect`),
  connectionStatus: (id) => client.get(`/whatsapp-numbers/${id}/status`)
}

export const projects = {
  list: () => client.get('/projects'),
  get: (id) => client.get(`/projects/${id}`),
  create: (data) => client.post('/projects', data),
  update: (id, data) => client.put(`/projects/${id}`, data),
  delete: (id) => client.delete(`/projects/${id}`),
  regenerateWebhookSecret: (id) => client.post(`/projects/${id}/regenerate-webhook-secret`),
  getWithApiKeys: (id) => client.get(`/projects/${id}/with-api-keys`)
}

export const apiKeys = {
  list: () => client.get('/api-keys'),
  get: (id) => client.get(`/api-keys/${id}`),
  create: (data) => client.post('/api-keys', data),
  update: (id, data) => client.put(`/api-keys/${id}`, data),
  delete: (id) => client.delete(`/api-keys/${id}`),
  regenerateSecret: (id) => client.post(`/api-keys/${id}/regenerate-secret`),
  setIpWhitelist: (id, ips) => client.post(`/api-keys/${id}/ip-whitelist`, { ips })
}

export const messages = {
  list: (params) => client.get('/messages', { params }),
  get: (id) => client.get(`/messages/${id}`),
  stats: (params) => client.get('/messages/stats', { params }),
  export: (params) => client.get('/messages/export', { params, responseType: 'blob' }),
  conversation: (projectId, contact, limit) => client.get(`/messages/conversation/${projectId}`, { params: { contact, limit } })
}

export const billing = {
  packages: () => client.get('/billing/packages'),
  subscription: () => client.get('/billing/subscription'),
  subscribe: (data) => client.post('/billing/subscribe', data),
  cancel: () => client.post('/billing/cancel'),
  usage: (params) => client.get('/billing/usage', { params }),
  quota: () => client.get('/billing/quota'),
  credits: () => client.get('/billing/credits'),
  invoices: () => client.get('/billing/invoices'),
  invoiceDetails: (id) => client.get(`/billing/invoices/${id}`)
}

export const webhooks = {
  logs: (params) => client.get('/webhooks/logs', { params }),
  stats: () => client.get('/webhooks/stats')
}

export const admin = {
  users: {
    list: (params) => client.get('/admin/users', { params }),
    get: (id) => client.get(`/admin/users/${id}`),
    create: (data) => client.post('/admin/users', data),
    update: (id, data) => client.put(`/admin/users/${id}`, data),
    suspend: (id) => client.post(`/admin/users/${id}/suspend`),
    ban: (id) => client.post(`/admin/users/${id}/ban`),
    activate: (id) => client.post(`/admin/users/${id}/activate`),
    resetPassword: (id) => client.post(`/admin/users/${id}/reset-password`),
    disconnectWhatsapp: (userId, numberId) => client.post(`/admin/users/${userId}/whatsapp/${numberId}/disconnect`)
  },
  packages: {
    list: () => client.get('/admin/packages'),
    get: (id) => client.get(`/admin/packages/${id}`),
    create: (data) => client.post('/admin/packages', data),
    update: (id, data) => client.put(`/admin/packages/${id}`, data),
    delete: (id) => client.delete(`/admin/packages/${id}`),
    setDefault: (id) => client.post(`/admin/packages/${id}/set-default`)
  },
  settings: {
    list: () => client.get('/admin/settings'),
    update: (data) => client.put('/admin/settings', data),
    routes: () => client.get('/admin/settings/routes'),
    updateRoute: (data) => client.put('/admin/settings/routes', data),
    enableRoute: (name) => client.post(`/admin/settings/routes/${name}/enable`),
    disableRoute: (name) => client.post(`/admin/settings/routes/${name}/disable`)
  },
  analytics: {
    dashboard: () => client.get('/admin/analytics/dashboard'),
    usage: (params) => client.get('/admin/analytics/usage', { params }),
    revenue: () => client.get('/admin/analytics/revenue'),
    logs: (params) => client.get('/admin/analytics/logs', { params }),
    topUsers: () => client.get('/admin/analytics/top-users'),
    messageStats: () => client.get('/admin/analytics/messages')
  }
}
