import { defineStore } from 'pinia'
import { auth } from '@/api/client'

export const useAuthStore = defineStore('auth', {
  state: () => ({
    user: JSON.parse(localStorage.getItem('user')) || null,
    token: localStorage.getItem('token') || null,
    loading: false,
    error: null
  }),
  
  getters: {
    isAuthenticated: (state) => !!state.token,
    isAdmin: (state) => state.user?.role === 'admin' || state.user?.role === 'superadmin',
    isSuperAdmin: (state) => state.user?.role === 'superadmin'
  },
  
  actions: {
    async login(email, password) {
      this.loading = true
      this.error = null
      
      try {
        const response = await auth.login({ email, password })
        const { user, token } = response.data.data
        
        this.user = user
        this.token = token
        
        localStorage.setItem('user', JSON.stringify(user))
        localStorage.setItem('token', token)
        
        return { success: true }
      } catch (error) {
        this.error = error.response?.data?.error || 'Login failed'
        return { success: false, error: this.error }
      } finally {
        this.loading = false
      }
    },
    
    async register(data) {
      this.loading = true
      this.error = null
      
      try {
        const response = await auth.register(data)
        const { user, token } = response.data.data
        
        this.user = user
        this.token = token
        
        localStorage.setItem('user', JSON.stringify(user))
        localStorage.setItem('token', token)
        
        return { success: true }
      } catch (error) {
        this.error = error.response?.data?.error || 'Registration failed'
        return { success: false, error: this.error }
      } finally {
        this.loading = false
      }
    },
    
    async googleLogin(code) {
      this.loading = true
      this.error = null
      
      try {
        const response = await auth.googleCallback(code)
        const { user, token } = response.data.data
        
        this.user = user
        this.token = token
        
        localStorage.setItem('user', JSON.stringify(user))
        localStorage.setItem('token', token)
        
        return { success: true }
      } catch (error) {
        this.error = error.response?.data?.error || 'Google login failed'
        return { success: false, error: this.error }
      } finally {
        this.loading = false
      }
    },
    
    async logout() {
      try {
        await auth.logout()
      } catch (error) {
        console.error('Logout error:', error)
      } finally {
        this.user = null
        this.token = null
        localStorage.removeItem('user')
        localStorage.removeItem('token')
      }
    },
    
    async checkAuth() {
      if (!this.token) return
      
      try {
        const response = await auth.me()
        this.user = response.data.data.user
        localStorage.setItem('user', JSON.stringify(this.user))
      } catch (error) {
        this.logout()
      }
    },
    
    async updateProfile(data) {
      this.loading = true
      
      try {
        const response = await auth.updateProfile(data)
        this.user = response.data.data.user
        localStorage.setItem('user', JSON.stringify(this.user))
        return { success: true }
      } catch (error) {
        return { success: false, error: error.response?.data?.error || 'Update failed' }
      } finally {
        this.loading = false
      }
    },
    
    async changePassword(data) {
      this.loading = true
      
      try {
        await auth.changePassword(data)
        return { success: true }
      } catch (error) {
        return { success: false, error: error.response?.data?.error || 'Password change failed' }
      } finally {
        this.loading = false
      }
    }
  }
})
