require('dotenv').config();

const express = require('express');
const cors = require('cors');
const pino = require('pino');

const routes = require('./routes');
const { initializeDatabase } = require('./utils/database');
const { BaileysManager } = require('./services/BaileysManager');

const logger = pino({
    level: process.env.LOG_LEVEL || 'info',
    transport: {
        target: 'pino-pretty',
        options: { colorize: true }
    }
});

const app = express();
const PORT = process.env.PORT || 3000;

app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true }));

app.use((req, res, next) => {
    logger.info({ method: req.method, url: req.url }, 'Incoming request');
    next();
});

const authMiddleware = (req, res, next) => {
    const gatewaySecret = req.headers['x-gateway-secret'];
    const expectedSecret = process.env.GATEWAY_SECRET;
    
    const publicPaths = ['/health', '/'];
    if (publicPaths.includes(req.path)) {
        return next();
    }
    
    if (!gatewaySecret || gatewaySecret !== expectedSecret) {
        return res.status(401).json({ success: false, error: 'Unauthorized' });
    }
    
    next();
};

app.use(authMiddleware);

app.get('/', (req, res) => {
    res.json({ 
        service: 'WhatsApp Gateway',
        version: '1.0.0',
        status: 'running'
    });
});

app.get('/health', (req, res) => {
    res.json({ 
        status: 'healthy',
        uptime: process.uptime(),
        timestamp: new Date().toISOString()
    });
});

app.use('/', routes);

app.use((err, req, res, next) => {
    logger.error({ err }, 'Unhandled error');
    res.status(500).json({ 
        success: false, 
        error: process.env.NODE_ENV === 'development' ? err.message : 'Internal server error'
    });
});

app.use((req, res) => {
    res.status(404).json({ success: false, error: 'Route not found' });
});

async function startServer() {
    try {
        await initializeDatabase();
        logger.info('Database connection established');
        
        const baileysManager = BaileysManager.getInstance();
        await baileysManager.restoreActiveSessions();
        logger.info('Baileys sessions restored');
        
        app.listen(PORT, () => {
            logger.info(`Gateway server running on port ${PORT}`);
        });
        
    } catch (error) {
        logger.error({ error }, 'Failed to start server');
        process.exit(1);
    }
}

process.on('SIGTERM', async () => {
    logger.info('SIGTERM received, shutting down gracefully');
    const baileysManager = BaileysManager.getInstance();
    await baileysManager.disconnectAll();
    process.exit(0);
});

process.on('SIGINT', async () => {
    logger.info('SIGINT received, shutting down gracefully');
    const baileysManager = BaileysManager.getInstance();
    await baileysManager.disconnectAll();
    process.exit(0);
});

startServer();

module.exports = app;
