const express = require('express');
const router = express.Router();

const { BaileysManager } = require('../services/BaileysManager');
const RoutingEngine = require('../services/RoutingEngine');
const { getWhatsappNumber, getNumberByQrToken } = require('../utils/database');

router.post('/send', async (req, res) => {
    try {
        const { message_id, channel, from, to, type, content, whatsapp_number_id } = req.body;
        
        if (!to || !whatsapp_number_id) {
            return res.status(400).json({ 
                success: false, 
                error: 'Missing required fields' 
            });
        }
        
        const result = await RoutingEngine.sendMessage({
            whatsapp_number_id,
            to,
            type: type || 'text',
            content,
            channel
        });
        
        res.json(result);
        
    } catch (error) {
        res.status(500).json({ 
            success: false, 
            error: error.message,
            error_code: 'SEND_ERROR'
        });
    }
});

router.post('/send-media', async (req, res) => {
    try {
        const { message_id, channel, from, to, type, media_url, caption, filename, whatsapp_number_id } = req.body;
        
        if (!to || !media_url || !whatsapp_number_id) {
            return res.status(400).json({ 
                success: false, 
                error: 'Missing required fields' 
            });
        }
        
        const result = await RoutingEngine.sendMediaMessage({
            whatsapp_number_id,
            to,
            type: type || 'image',
            media_url,
            caption,
            filename,
            channel
        });
        
        res.json(result);
        
    } catch (error) {
        res.status(500).json({ 
            success: false, 
            error: error.message,
            error_code: 'SEND_ERROR'
        });
    }
});

router.post('/request-qr', async (req, res) => {
    try {
        const { user_id, whatsapp_number_id } = req.body;
        
        if (!whatsapp_number_id) {
            return res.status(400).json({ 
                success: false, 
                error: 'WhatsApp number ID required' 
            });
        }
        
        const number = await getWhatsappNumber(whatsapp_number_id);
        
        if (!number) {
            return res.status(404).json({ 
                success: false, 
                error: 'WhatsApp number not found' 
            });
        }
        
        if (!['baileys', 'hybrid'].includes(number.channel)) {
            return res.status(400).json({ 
                success: false, 
                error: 'QR code only available for Baileys channel' 
            });
        }
        
        const baileysManager = BaileysManager.getInstance();
        await baileysManager.createSession(whatsapp_number_id);
        
        await new Promise(resolve => setTimeout(resolve, 2000));
        
        const updatedNumber = await getWhatsappNumber(whatsapp_number_id);
        
        res.json({
            success: true,
            qr_token: updatedNumber.baileys_qr_token
        });
        
    } catch (error) {
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

router.get('/qr-status', async (req, res) => {
    try {
        const { token } = req.query;
        
        if (!token) {
            return res.status(400).json({ 
                success: false, 
                error: 'QR token required' 
            });
        }
        
        const baileysManager = BaileysManager.getInstance();
        const qrData = baileysManager.getQrCode(token);
        
        res.json(qrData);
        
    } catch (error) {
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

router.post('/disconnect', async (req, res) => {
    try {
        const { whatsapp_number_id } = req.body;
        
        if (!whatsapp_number_id) {
            return res.status(400).json({ 
                success: false, 
                error: 'WhatsApp number ID required' 
            });
        }
        
        const baileysManager = BaileysManager.getInstance();
        await baileysManager.disconnect(whatsapp_number_id);
        
        res.json({ success: true });
        
    } catch (error) {
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

router.get('/connection-status', async (req, res) => {
    try {
        const { whatsapp_number_id } = req.query;
        
        if (!whatsapp_number_id) {
            return res.status(400).json({ 
                success: false, 
                error: 'WhatsApp number ID required' 
            });
        }
        
        const baileysManager = BaileysManager.getInstance();
        const status = baileysManager.getConnectionStatus(parseInt(whatsapp_number_id));
        
        res.json(status);
        
    } catch (error) {
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

router.get('/sessions', async (req, res) => {
    try {
        const baileysManager = BaileysManager.getInstance();
        const sessions = [];
        
        for (const [id, session] of baileysManager.sessions) {
            sessions.push({
                whatsapp_number_id: id,
                phone_number: session.phoneNumber,
                is_connected: session.isConnected,
                retries: session.retries
            });
        }
        
        res.json({ success: true, sessions });
        
    } catch (error) {
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

module.exports = router;
