const { default: makeWASocket, DisconnectReason, useMultiFileAuthState, fetchLatestBaileysVersion } = require('@whiskeysockets/baileys');
const pino = require('pino');
const fs = require('fs');
const path = require('path');
const QRCode = require('qrcode');

const { generateToken, formatPhoneForWhatsapp, extractPhoneFromJid, getMessageType, extractMessageContent } = require('../utils/helpers');
const { getWhatsappNumber, updateWhatsappNumberStatus, setQrToken, clearQrToken, setBaileysConnected, getConnectedBaileysNumbers } = require('../utils/database');
const WebhookService = require('./WebhookService');

const logger = pino({ level: process.env.LOG_LEVEL || 'info' });

class BaileysManager {
    static instance = null;
    
    constructor() {
        this.sessions = new Map();
        this.qrCodes = new Map();
        this.sessionsPath = process.env.BAILEYS_SESSIONS_PATH || './sessions';
        this.reconnectInterval = parseInt(process.env.BAILEYS_RECONNECT_INTERVAL) || 5000;
        this.maxRetries = parseInt(process.env.BAILEYS_MAX_RETRIES) || 5;
        
        if (!fs.existsSync(this.sessionsPath)) {
            fs.mkdirSync(this.sessionsPath, { recursive: true });
        }
    }
    
    static getInstance() {
        if (!BaileysManager.instance) {
            BaileysManager.instance = new BaileysManager();
        }
        return BaileysManager.instance;
    }
    
    async createSession(whatsappNumberId) {
        const numberData = await getWhatsappNumber(whatsappNumberId);
        
        if (!numberData) {
            throw new Error('WhatsApp number not found');
        }
        
        if (this.sessions.has(whatsappNumberId)) {
            logger.info({ whatsappNumberId }, 'Session already exists');
            return this.sessions.get(whatsappNumberId);
        }
        
        const sessionPath = path.join(this.sessionsPath, `session_${whatsappNumberId}`);
        
        if (!fs.existsSync(sessionPath)) {
            fs.mkdirSync(sessionPath, { recursive: true });
        }
        
        const { state, saveCreds } = await useMultiFileAuthState(sessionPath);
        const { version } = await fetchLatestBaileysVersion();
        
        const sock = makeWASocket({
            version,
            auth: state,
            printQRInTerminal: false,
            logger: pino({ level: 'silent' }),
            browser: ['WhatsApp Gateway', 'Chrome', '120.0.0'],
            connectTimeoutMs: 60000,
            defaultQueryTimeoutMs: 60000,
            keepAliveIntervalMs: 30000,
            emitOwnEvents: true,
            markOnlineOnConnect: true
        });
        
        const sessionData = {
            socket: sock,
            whatsappNumberId,
            phoneNumber: numberData.phone_number,
            userId: numberData.user_id,
            retries: 0,
            isConnected: false
        };
        
        this.sessions.set(whatsappNumberId, sessionData);
        
        sock.ev.on('creds.update', saveCreds);
        
        sock.ev.on('connection.update', async (update) => {
            await this.handleConnectionUpdate(whatsappNumberId, update);
        });
        
        sock.ev.on('messages.upsert', async (m) => {
            await this.handleIncomingMessage(whatsappNumberId, m);
        });
        
        sock.ev.on('messages.update', async (updates) => {
            await this.handleMessageStatusUpdate(whatsappNumberId, updates);
        });
        
        logger.info({ whatsappNumberId }, 'Baileys session created');
        
        return sessionData;
    }
    
    async handleConnectionUpdate(whatsappNumberId, update) {
        const { connection, lastDisconnect, qr } = update;
        const session = this.sessions.get(whatsappNumberId);
        
        if (!session) return;
        
        if (qr) {
            const qrToken = generateToken(16);
            const qrDataUrl = await QRCode.toDataURL(qr);
            
            this.qrCodes.set(qrToken, {
                whatsappNumberId,
                qr: qrDataUrl,
                createdAt: Date.now(),
                status: 'pending'
            });
            
            await setQrToken(whatsappNumberId, qrToken);
            
            logger.info({ whatsappNumberId, qrToken }, 'QR code generated');
            
            await WebhookService.notifyBackend('qr.update', {
                whatsapp_number_id: whatsappNumberId,
                qr_token: qrToken,
                qr_data: qrDataUrl
            });
        }
        
        if (connection === 'close') {
            const statusCode = lastDisconnect?.error?.output?.statusCode;
            const shouldReconnect = statusCode !== DisconnectReason.loggedOut;
            
            logger.info({ whatsappNumberId, statusCode, shouldReconnect }, 'Connection closed');
            
            session.isConnected = false;
            await updateWhatsappNumberStatus(whatsappNumberId, 'disconnected');
            
            await WebhookService.notifyBackend('connection.update', {
                whatsapp_number_id: whatsappNumberId,
                status: 'disconnected',
                reason: statusCode
            });
            
            if (shouldReconnect && session.retries < this.maxRetries) {
                session.retries++;
                logger.info({ whatsappNumberId, retry: session.retries }, 'Attempting reconnect');
                
                setTimeout(async () => {
                    this.sessions.delete(whatsappNumberId);
                    await this.createSession(whatsappNumberId);
                }, this.reconnectInterval * session.retries);
            } else {
                this.sessions.delete(whatsappNumberId);
                
                if (statusCode === DisconnectReason.loggedOut) {
                    const sessionPath = path.join(this.sessionsPath, `session_${whatsappNumberId}`);
                    if (fs.existsSync(sessionPath)) {
                        fs.rmSync(sessionPath, { recursive: true });
                    }
                }
            }
        }
        
        if (connection === 'open') {
            session.isConnected = true;
            session.retries = 0;
            
            const sessionId = `session_${whatsappNumberId}`;
            await setBaileysConnected(whatsappNumberId, sessionId);
            await clearQrToken(whatsappNumberId);
            
            for (const [token, data] of this.qrCodes.entries()) {
                if (data.whatsappNumberId === whatsappNumberId) {
                    data.status = 'connected';
                }
            }
            
            logger.info({ whatsappNumberId }, 'Connection established');
            
            await WebhookService.notifyBackend('connection.update', {
                whatsapp_number_id: whatsappNumberId,
                status: 'connected'
            });
        }
    }
    
    async handleIncomingMessage(whatsappNumberId, { messages, type }) {
        if (type !== 'notify') return;
        
        const session = this.sessions.get(whatsappNumberId);
        if (!session) return;
        
        for (const msg of messages) {
            if (msg.key.fromMe) continue;
            
            const fromNumber = extractPhoneFromJid(msg.key.remoteJid);
            const messageType = getMessageType(msg.message);
            const content = extractMessageContent(msg.message);
            
            logger.info({ whatsappNumberId, from: fromNumber, type: messageType }, 'Message received');
            
            await WebhookService.notifyBackend('message', {
                whatsapp_number_id: whatsappNumberId,
                message: {
                    id: msg.key.id,
                    from: fromNumber,
                    to: session.phoneNumber,
                    type: messageType,
                    content: content,
                    timestamp: msg.messageTimestamp,
                    raw: msg.message
                }
            });
        }
    }
    
    async handleMessageStatusUpdate(whatsappNumberId, updates) {
        for (const update of updates) {
            if (!update.update?.status) continue;
            
            const statusMap = {
                2: 'sent',
                3: 'delivered',
                4: 'read'
            };
            
            const status = statusMap[update.update.status];
            if (!status) continue;
            
            logger.info({ whatsappNumberId, messageId: update.key.id, status }, 'Message status update');
            
            await WebhookService.notifyBackend('message.status', {
                whatsapp_number_id: whatsappNumberId,
                message_id: update.key.id,
                status: status
            });
        }
    }
    
    async sendTextMessage(whatsappNumberId, to, message) {
        const session = this.sessions.get(whatsappNumberId);
        
        if (!session || !session.isConnected) {
            return {
                success: false,
                error: 'Session not connected',
                error_code: 'NOT_CONNECTED'
            };
        }
        
        try {
            const jid = formatPhoneForWhatsapp(to);
            const result = await session.socket.sendMessage(jid, { text: message });
            
            logger.info({ whatsappNumberId, to, messageId: result.key.id }, 'Baileys message sent');
            
            return {
                success: true,
                external_id: result.key.id,
                channel: 'baileys'
            };
            
        } catch (error) {
            logger.error({ error: error.message, whatsappNumberId, to }, 'Baileys send failed');
            
            return {
                success: false,
                error: error.message,
                error_code: 'SEND_FAILED'
            };
        }
    }
    
    async sendMediaMessage(whatsappNumberId, to, mediaType, mediaUrl, caption = null, filename = null) {
        const session = this.sessions.get(whatsappNumberId);
        
        if (!session || !session.isConnected) {
            return {
                success: false,
                error: 'Session not connected',
                error_code: 'NOT_CONNECTED'
            };
        }
        
        try {
            const jid = formatPhoneForWhatsapp(to);
            let messageContent = {};
            
            switch (mediaType) {
                case 'image':
                    messageContent = { image: { url: mediaUrl }, caption };
                    break;
                case 'video':
                    messageContent = { video: { url: mediaUrl }, caption };
                    break;
                case 'audio':
                    messageContent = { audio: { url: mediaUrl }, mimetype: 'audio/mpeg' };
                    break;
                case 'document':
                    messageContent = { 
                        document: { url: mediaUrl }, 
                        fileName: filename || 'document',
                        caption 
                    };
                    break;
                default:
                    messageContent = { image: { url: mediaUrl }, caption };
            }
            
            const result = await session.socket.sendMessage(jid, messageContent);
            
            logger.info({ whatsappNumberId, to, mediaType, messageId: result.key.id }, 'Baileys media sent');
            
            return {
                success: true,
                external_id: result.key.id,
                channel: 'baileys'
            };
            
        } catch (error) {
            logger.error({ error: error.message, whatsappNumberId, to, mediaType }, 'Baileys media send failed');
            
            return {
                success: false,
                error: error.message,
                error_code: 'SEND_FAILED'
            };
        }
    }
    
    async disconnect(whatsappNumberId) {
        const session = this.sessions.get(whatsappNumberId);
        
        if (session) {
            try {
                await session.socket.logout();
            } catch (error) {
                logger.error({ error: error.message, whatsappNumberId }, 'Logout error');
            }
            
            session.socket.end();
            this.sessions.delete(whatsappNumberId);
            
            await updateWhatsappNumberStatus(whatsappNumberId, 'disconnected');
            
            const sessionPath = path.join(this.sessionsPath, `session_${whatsappNumberId}`);
            if (fs.existsSync(sessionPath)) {
                fs.rmSync(sessionPath, { recursive: true });
            }
            
            logger.info({ whatsappNumberId }, 'Session disconnected and removed');
        }
        
        return { success: true };
    }
    
    async disconnectAll() {
        for (const [whatsappNumberId, session] of this.sessions) {
            try {
                session.socket.end();
                await updateWhatsappNumberStatus(whatsappNumberId, 'disconnected');
            } catch (error) {
                logger.error({ error: error.message, whatsappNumberId }, 'Disconnect error');
            }
        }
        
        this.sessions.clear();
        logger.info('All sessions disconnected');
    }
    
    async restoreActiveSessions() {
        try {
            const connectedNumbers = await getConnectedBaileysNumbers();
            
            for (const number of connectedNumbers) {
                const sessionPath = path.join(this.sessionsPath, `session_${number.id}`);
                
                if (fs.existsSync(sessionPath)) {
                    logger.info({ whatsappNumberId: number.id }, 'Restoring session');
                    await this.createSession(number.id);
                }
            }
            
            logger.info({ count: connectedNumbers.length }, 'Sessions restoration complete');
            
        } catch (error) {
            logger.error({ error: error.message }, 'Failed to restore sessions');
        }
    }
    
    getQrCode(token) {
        const qrData = this.qrCodes.get(token);
        
        if (!qrData) {
            return { success: false, status: 'not_found' };
        }
        
        if (Date.now() - qrData.createdAt > 120000) {
            this.qrCodes.delete(token);
            return { success: false, status: 'expired' };
        }
        
        return {
            success: true,
            status: qrData.status,
            qr: qrData.status === 'pending' ? qrData.qr : null
        };
    }
    
    getConnectionStatus(whatsappNumberId) {
        const session = this.sessions.get(whatsappNumberId);
        
        return {
            connected: session?.isConnected || false,
            status: session?.isConnected ? 'connected' : 'disconnected'
        };
    }
    
    isConnected(whatsappNumberId) {
        const session = this.sessions.get(whatsappNumberId);
        return session?.isConnected || false;
    }
}

module.exports = { BaileysManager };
