const axios = require('axios');
const pino = require('pino');

const logger = pino({ level: process.env.LOG_LEVEL || 'info' });

class MetaService {
    constructor() {
        this.apiVersion = process.env.META_API_VERSION || 'v18.0';
        this.baseUrl = `https://graph.facebook.com/${this.apiVersion}`;
    }
    
    async sendTextMessage(phoneNumberId, accessToken, to, message) {
        try {
            const response = await axios.post(
                `${this.baseUrl}/${phoneNumberId}/messages`,
                {
                    messaging_product: 'whatsapp',
                    recipient_type: 'individual',
                    to: to.replace(/[^0-9]/g, ''),
                    type: 'text',
                    text: { body: message }
                },
                {
                    headers: {
                        'Authorization': `Bearer ${accessToken}`,
                        'Content-Type': 'application/json'
                    }
                }
            );
            
            logger.info({ to, messageId: response.data.messages?.[0]?.id }, 'Meta message sent');
            
            return {
                success: true,
                external_id: response.data.messages?.[0]?.id,
                channel: 'meta'
            };
            
        } catch (error) {
            logger.error({ error: error.response?.data || error.message }, 'Meta send failed');
            
            return {
                success: false,
                error: error.response?.data?.error?.message || error.message,
                error_code: error.response?.data?.error?.code || 'UNKNOWN'
            };
        }
    }
    
    async sendMediaMessage(phoneNumberId, accessToken, to, mediaType, mediaUrl, caption = null) {
        try {
            const mediaPayload = {
                link: mediaUrl
            };
            
            if (caption && ['image', 'video', 'document'].includes(mediaType)) {
                mediaPayload.caption = caption;
            }
            
            const response = await axios.post(
                `${this.baseUrl}/${phoneNumberId}/messages`,
                {
                    messaging_product: 'whatsapp',
                    recipient_type: 'individual',
                    to: to.replace(/[^0-9]/g, ''),
                    type: mediaType,
                    [mediaType]: mediaPayload
                },
                {
                    headers: {
                        'Authorization': `Bearer ${accessToken}`,
                        'Content-Type': 'application/json'
                    }
                }
            );
            
            logger.info({ to, mediaType, messageId: response.data.messages?.[0]?.id }, 'Meta media sent');
            
            return {
                success: true,
                external_id: response.data.messages?.[0]?.id,
                channel: 'meta'
            };
            
        } catch (error) {
            logger.error({ error: error.response?.data || error.message }, 'Meta media send failed');
            
            return {
                success: false,
                error: error.response?.data?.error?.message || error.message,
                error_code: error.response?.data?.error?.code || 'UNKNOWN'
            };
        }
    }
    
    async sendTemplateMessage(phoneNumberId, accessToken, to, templateName, languageCode, components = []) {
        try {
            const response = await axios.post(
                `${this.baseUrl}/${phoneNumberId}/messages`,
                {
                    messaging_product: 'whatsapp',
                    recipient_type: 'individual',
                    to: to.replace(/[^0-9]/g, ''),
                    type: 'template',
                    template: {
                        name: templateName,
                        language: { code: languageCode },
                        components: components
                    }
                },
                {
                    headers: {
                        'Authorization': `Bearer ${accessToken}`,
                        'Content-Type': 'application/json'
                    }
                }
            );
            
            logger.info({ to, templateName, messageId: response.data.messages?.[0]?.id }, 'Meta template sent');
            
            return {
                success: true,
                external_id: response.data.messages?.[0]?.id,
                channel: 'meta'
            };
            
        } catch (error) {
            logger.error({ error: error.response?.data || error.message }, 'Meta template send failed');
            
            return {
                success: false,
                error: error.response?.data?.error?.message || error.message,
                error_code: error.response?.data?.error?.code || 'UNKNOWN'
            };
        }
    }
    
    async getMediaUrl(mediaId, accessToken) {
        try {
            const response = await axios.get(
                `${this.baseUrl}/${mediaId}`,
                {
                    headers: {
                        'Authorization': `Bearer ${accessToken}`
                    }
                }
            );
            
            return response.data.url;
            
        } catch (error) {
            logger.error({ error: error.message, mediaId }, 'Failed to get media URL');
            return null;
        }
    }
    
    async downloadMedia(mediaUrl, accessToken) {
        try {
            const response = await axios.get(mediaUrl, {
                headers: {
                    'Authorization': `Bearer ${accessToken}`
                },
                responseType: 'arraybuffer'
            });
            
            return {
                data: response.data,
                contentType: response.headers['content-type']
            };
            
        } catch (error) {
            logger.error({ error: error.message }, 'Failed to download media');
            return null;
        }
    }
    
    async markAsRead(phoneNumberId, accessToken, messageId) {
        try {
            await axios.post(
                `${this.baseUrl}/${phoneNumberId}/messages`,
                {
                    messaging_product: 'whatsapp',
                    status: 'read',
                    message_id: messageId
                },
                {
                    headers: {
                        'Authorization': `Bearer ${accessToken}`,
                        'Content-Type': 'application/json'
                    }
                }
            );
            
            return true;
            
        } catch (error) {
            logger.error({ error: error.message, messageId }, 'Failed to mark as read');
            return false;
        }
    }
}

module.exports = new MetaService();
