const pino = require('pino');
const MetaService = require('./MetaService');
const { BaileysManager } = require('./BaileysManager');
const { getWhatsappNumber, query } = require('../utils/database');

const logger = pino({ level: process.env.LOG_LEVEL || 'info' });

class RoutingEngine {
    async getEnabledRoutes() {
        const routes = await query(
            'SELECT * FROM route_settings WHERE is_enabled = 1 ORDER BY priority ASC'
        );
        return routes;
    }
    
    async determineChannel(whatsappNumberId, preferredChannel = null) {
        const number = await getWhatsappNumber(whatsappNumberId);
        
        if (!number) {
            throw new Error('WhatsApp number not found');
        }
        
        const enabledRoutes = await this.getEnabledRoutes();
        const enabledChannels = enabledRoutes.map(r => r.route_name);
        
        if (preferredChannel && enabledChannels.includes(preferredChannel)) {
            if (preferredChannel === 'baileys') {
                const baileysManager = BaileysManager.getInstance();
                if (baileysManager.isConnected(whatsappNumberId)) {
                    return 'baileys';
                }
            } else if (preferredChannel === 'meta' && number.meta_phone_number_id) {
                return 'meta';
            }
        }
        
        if (number.channel === 'meta' && enabledChannels.includes('meta') && number.meta_phone_number_id) {
            return 'meta';
        }
        
        if (['baileys', 'hybrid'].includes(number.channel) && enabledChannels.includes('baileys')) {
            const baileysManager = BaileysManager.getInstance();
            if (baileysManager.isConnected(whatsappNumberId)) {
                return 'baileys';
            }
        }
        
        if (number.channel === 'hybrid') {
            if (enabledChannels.includes('meta') && number.meta_phone_number_id) {
                return 'meta';
            }
        }
        
        throw new Error('No available channel for sending');
    }
    
    async sendMessage(messageData) {
        const { whatsapp_number_id, to, type, content, channel: preferredChannel } = messageData;
        
        const channel = await this.determineChannel(whatsapp_number_id, preferredChannel);
        const number = await getWhatsappNumber(whatsapp_number_id);
        
        logger.info({ whatsapp_number_id, to, channel, type }, 'Routing message');
        
        if (channel === 'meta') {
            return await this.sendViaMeta(number, to, type, content);
        } else {
            return await this.sendViaBaileys(whatsapp_number_id, to, type, content);
        }
    }
    
    async sendViaMeta(number, to, type, content) {
        const accessToken = this.decryptToken(number.meta_access_token);
        
        if (!accessToken || !number.meta_phone_number_id) {
            return {
                success: false,
                error: 'Meta API not configured',
                error_code: 'META_NOT_CONFIGURED'
            };
        }
        
        if (type === 'text') {
            return await MetaService.sendTextMessage(
                number.meta_phone_number_id,
                accessToken,
                to,
                content
            );
        }
        
        return {
            success: false,
            error: 'Unsupported message type for Meta',
            error_code: 'UNSUPPORTED_TYPE'
        };
    }
    
    async sendViaBaileys(whatsappNumberId, to, type, content) {
        const baileysManager = BaileysManager.getInstance();
        
        if (type === 'text') {
            return await baileysManager.sendTextMessage(whatsappNumberId, to, content);
        }
        
        return {
            success: false,
            error: 'Unsupported message type for Baileys',
            error_code: 'UNSUPPORTED_TYPE'
        };
    }
    
    async sendMediaMessage(messageData) {
        const { whatsapp_number_id, to, type, media_url, caption, filename, channel: preferredChannel } = messageData;
        
        const channel = await this.determineChannel(whatsapp_number_id, preferredChannel);
        const number = await getWhatsappNumber(whatsapp_number_id);
        
        logger.info({ whatsapp_number_id, to, channel, type }, 'Routing media message');
        
        if (channel === 'meta') {
            const accessToken = this.decryptToken(number.meta_access_token);
            
            if (!accessToken || !number.meta_phone_number_id) {
                return {
                    success: false,
                    error: 'Meta API not configured',
                    error_code: 'META_NOT_CONFIGURED'
                };
            }
            
            return await MetaService.sendMediaMessage(
                number.meta_phone_number_id,
                accessToken,
                to,
                type,
                media_url,
                caption
            );
        } else {
            const baileysManager = BaileysManager.getInstance();
            return await baileysManager.sendMediaMessage(
                whatsapp_number_id,
                to,
                type,
                media_url,
                caption,
                filename
            );
        }
    }
    
    decryptToken(encryptedToken) {
        if (!encryptedToken) return null;
        
        try {
            const crypto = require('crypto');
            const key = process.env.APP_KEY || 'default-key';
            const data = Buffer.from(encryptedToken, 'base64');
            const iv = data.slice(0, 16);
            const encrypted = data.slice(16).toString();
            
            const decipher = crypto.createDecipheriv('aes-256-cbc', Buffer.from(key.padEnd(32).slice(0, 32)), iv);
            let decrypted = decipher.update(encrypted, 'base64', 'utf8');
            decrypted += decipher.final('utf8');
            
            return decrypted;
        } catch (error) {
            logger.error({ error: error.message }, 'Token decryption failed');
            return encryptedToken;
        }
    }
}

module.exports = new RoutingEngine();
