const axios = require('axios');
const pino = require('pino');

const logger = pino({ level: process.env.LOG_LEVEL || 'info' });

class WebhookService {
    constructor() {
        this.backendUrl = process.env.BACKEND_URL || 'http://localhost';
        this.gatewaySecret = process.env.GATEWAY_SECRET || '';
    }
    
    async notifyBackend(event, data) {
        try {
            const endpoint = this.getEndpointForEvent(event);
            
            const response = await axios.post(
                `${this.backendUrl}${endpoint}`,
                {
                    event,
                    ...data
                },
                {
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Gateway-Secret': this.gatewaySecret
                    },
                    timeout: 30000
                }
            );
            
            logger.info({ event, status: response.status }, 'Backend notified');
            
            return { success: true, data: response.data };
            
        } catch (error) {
            logger.error({ 
                event, 
                error: error.response?.data || error.message 
            }, 'Failed to notify backend');
            
            return { success: false, error: error.message };
        }
    }
    
    getEndpointForEvent(event) {
        const endpoints = {
            'message': '/webhooks/baileys',
            'message.status': '/webhooks/baileys',
            'connection.update': '/webhooks/baileys',
            'qr.update': '/webhooks/baileys'
        };
        
        return endpoints[event] || '/webhooks/baileys';
    }
    
    async sendToClientWebhook(webhookUrl, webhookSecret, payload) {
        try {
            const payloadString = JSON.stringify(payload);
            const crypto = require('crypto');
            const signature = crypto.createHmac('sha256', webhookSecret)
                .update(payloadString)
                .digest('hex');
            
            const response = await axios.post(webhookUrl, payload, {
                headers: {
                    'Content-Type': 'application/json',
                    'X-Signature': signature,
                    'X-Webhook-Event': payload.event || 'unknown',
                    'User-Agent': 'WhatsAppGateway/1.0'
                },
                timeout: 30000
            });
            
            logger.info({ 
                webhookUrl, 
                event: payload.event, 
                status: response.status 
            }, 'Client webhook delivered');
            
            return {
                success: true,
                status: response.status,
                response: response.data
            };
            
        } catch (error) {
            logger.error({ 
                webhookUrl, 
                error: error.response?.data || error.message 
            }, 'Client webhook failed');
            
            return {
                success: false,
                status: error.response?.status || 0,
                error: error.message
            };
        }
    }
}

module.exports = new WebhookService();
