const mysql = require('mysql2/promise');

let pool = null;

async function initializeDatabase() {
    pool = mysql.createPool({
        host: process.env.DB_HOST || 'localhost',
        port: process.env.DB_PORT || 3306,
        user: process.env.DB_USER || 'root',
        password: process.env.DB_PASSWORD || '',
        database: process.env.DB_NAME || 'whatsapp_gateway',
        waitForConnections: true,
        connectionLimit: 10,
        queueLimit: 0
    });
    
    const connection = await pool.getConnection();
    await connection.ping();
    connection.release();
    
    return pool;
}

function getPool() {
    if (!pool) {
        throw new Error('Database not initialized');
    }
    return pool;
}

async function query(sql, params = []) {
    const [rows] = await getPool().execute(sql, params);
    return rows;
}

async function getWhatsappNumber(numberId) {
    const rows = await query(
        'SELECT * FROM whatsapp_numbers WHERE id = ?',
        [numberId]
    );
    return rows[0] || null;
}

async function updateWhatsappNumberStatus(numberId, status) {
    await query(
        'UPDATE whatsapp_numbers SET status = ?, last_active_at = NOW() WHERE id = ?',
        [status, numberId]
    );
}

async function setQrToken(numberId, token) {
    await query(
        'UPDATE whatsapp_numbers SET baileys_qr_token = ?, status = ? WHERE id = ?',
        [token, 'pending', numberId]
    );
}

async function clearQrToken(numberId) {
    await query(
        'UPDATE whatsapp_numbers SET baileys_qr_token = NULL WHERE id = ?',
        [numberId]
    );
}

async function setBaileysConnected(numberId, sessionId) {
    await query(
        `UPDATE whatsapp_numbers 
         SET status = 'connected', baileys_session_id = ?, baileys_connected_at = NOW(), 
             baileys_qr_token = NULL, last_active_at = NOW() 
         WHERE id = ?`,
        [sessionId, numberId]
    );
}

async function getConnectedBaileysNumbers() {
    return await query(
        `SELECT * FROM whatsapp_numbers 
         WHERE status = 'connected' AND channel IN ('baileys', 'hybrid') AND is_active = 1`
    );
}

async function getNumberByQrToken(token) {
    const rows = await query(
        'SELECT * FROM whatsapp_numbers WHERE baileys_qr_token = ?',
        [token]
    );
    return rows[0] || null;
}

module.exports = {
    initializeDatabase,
    getPool,
    query,
    getWhatsappNumber,
    updateWhatsappNumberStatus,
    setQrToken,
    clearQrToken,
    setBaileysConnected,
    getConnectedBaileysNumbers,
    getNumberByQrToken
};
